# Copyright (C) 2006-2007 Red Hat, Inc.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from sugar3.graphics import style
from sugar3.graphics.icon import CanvasIcon

from jarabe.view.buddymenu import BuddyMenu
from jarabe.util.normalize import normalize_string


_FILTERED_ALPHA = 0.33


class BuddyIcon(CanvasIcon):

    def __init__(self, buddy, pixel_size=style.STANDARD_ICON_SIZE):
        CanvasIcon.__init__(self, icon_name='computer-xo',
                            pixel_size=pixel_size)

        self._filtered = False
        self._buddy = buddy
        self._buddy.connect('notify::present', self.__buddy_notify_present_cb)
        self._buddy.connect('notify::color', self.__buddy_notify_color_cb)

        self.palette_invoker.props.toggle_palette = True
        self.palette_invoker.cache_palette = False

        self._update_color()

    def create_palette(self):
        palette = BuddyMenu(self._buddy)
        self.connect_to_palette_pop_events(palette)
        return palette

    def __buddy_notify_present_cb(self, buddy, pspec):
        # Update the icon's color when the buddy comes and goes
        self._update_color()

    def __buddy_notify_color_cb(self, buddy, pspec):
        self._update_color()

    def _update_color(self):
        # keep the icon in the palette in sync with the view
        palette = self.get_palette()
        self.props.xo_color = self._buddy.get_color()
        if self._filtered:
            self.alpha = _FILTERED_ALPHA
            if palette is not None:
                palette.props.icon.props.stroke_color = self.props.stroke_color
                palette.props.icon.props.fill_color = self.props.fill_color
        else:
            self.alpha = 1.0
            if palette is not None:
                palette.props.icon.props.xo_color = self._buddy.get_color()

    def set_filter(self, query):
        if isinstance(self._buddy.get_nick(), bytes):
            nick = self._buddy.get_nick().decode()
        else:
            nick = self._buddy.get_nick()
        normalized_name = normalize_string(nick)
        self._filtered = (normalized_name.find(query) == -1) \
            and not self._buddy.is_owner()
        self._update_color()

    def get_positioning_data(self):
        return self._buddy.get_key()
