/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.DigestException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.CorruptRecordException;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.MutableRecordBatch;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.utils.BufferSupplier;
import org.apache.kafka.common.utils.CloseableIterator;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.storage.internals.log.AbortedTxn;
import org.apache.kafka.storage.internals.log.CleanedTransactionMetadata;
import org.apache.kafka.storage.internals.log.CleanerStats;
import org.apache.kafka.storage.internals.log.LastRecord;
import org.apache.kafka.storage.internals.log.LogCleaningAbortedException;
import org.apache.kafka.storage.internals.log.LogSegment;
import org.apache.kafka.storage.internals.log.LogSegmentOffsetOverflowException;
import org.apache.kafka.storage.internals.log.LogToClean;
import org.apache.kafka.storage.internals.log.OffsetMap;
import org.apache.kafka.storage.internals.log.UnifiedLog;
import org.apache.kafka.storage.internals.utils.Throttler;
import org.slf4j.Logger;

public class Cleaner {
    private final BufferSupplier decompressionBufferSupplier = BufferSupplier.create();
    private final Logger logger;
    private final int id;
    private final OffsetMap offsetMap;
    private final int ioBufferSize;
    private final int maxIoBufferSize;
    private final double dupBufferLoadFactor;
    private final Throttler throttler;
    private final Time time;
    private final Consumer<TopicPartition> checkDone;
    private ByteBuffer readBuffer;
    private ByteBuffer writeBuffer;

    public Cleaner(int id, OffsetMap offsetMap, int ioBufferSize, int maxIoBufferSize, double dupBufferLoadFactor, Throttler throttler, Time time, Consumer<TopicPartition> checkDone) {
        this.id = id;
        this.offsetMap = offsetMap;
        this.ioBufferSize = ioBufferSize;
        this.maxIoBufferSize = maxIoBufferSize;
        this.dupBufferLoadFactor = dupBufferLoadFactor;
        this.throttler = throttler;
        this.time = time;
        this.checkDone = checkDone;
        this.logger = new LogContext("Cleaner " + id + ": ").logger(Cleaner.class);
        this.readBuffer = ByteBuffer.allocate(ioBufferSize);
        this.writeBuffer = ByteBuffer.allocate(ioBufferSize);
        assert ((double)offsetMap.slots() * dupBufferLoadFactor > 1.0) : "offset map is too small to fit in even a single message, so log cleaning will never make progress. You can increase log.cleaner.dedupe.buffer.size or decrease log.cleaner.threads";
    }

    public int id() {
        return this.id;
    }

    public OffsetMap offsetMap() {
        return this.offsetMap;
    }

    public Map.Entry<Long, CleanerStats> clean(LogToClean cleanable) throws IOException, DigestException {
        return this.doClean(cleanable, this.time.milliseconds());
    }

    public Map.Entry<Long, CleanerStats> doClean(LogToClean cleanable, long currentTime) throws IOException, DigestException {
        UnifiedLog log = cleanable.log();
        this.logger.info("Beginning cleaning of log {}", (Object)log.name());
        List<LogSegment> segments = log.logSegments(0L, cleanable.firstDirtyOffset());
        long legacyDeleteHorizonMs = segments.isEmpty() ? 0L : segments.get(segments.size() - 1).lastModified() - log.config().deleteRetentionMs;
        CleanerStats stats = new CleanerStats(Time.SYSTEM);
        this.logger.info("Building offset map for {}...", (Object)log.name());
        long upperBoundOffset = cleanable.firstUncleanableOffset();
        this.buildOffsetMap(log, cleanable.firstDirtyOffset(), upperBoundOffset, this.offsetMap, stats);
        long endOffset = this.offsetMap.latestOffset() + 1L;
        stats.indexDone();
        segments = log.logSegments(0L, cleanable.firstUncleanableOffset());
        long cleanableHorizonMs = segments.isEmpty() ? 0L : segments.get(segments.size() - 1).lastModified();
        this.logger.info("Cleaning log {} (cleaning prior to {}, discarding tombstones prior to upper bound deletion horizon {})...", new Object[]{log.name(), new Date(cleanableHorizonMs), new Date(legacyDeleteHorizonMs)});
        CleanedTransactionMetadata transactionMetadata = new CleanedTransactionMetadata();
        List<List<LogSegment>> groupedSegments = this.groupSegmentsBySize(log.logSegments(0L, endOffset), log.config().segmentSize(), log.config().maxIndexSize, cleanable.firstUncleanableOffset());
        for (List<LogSegment> group : groupedSegments) {
            this.cleanSegments(log, group, this.offsetMap, currentTime, stats, transactionMetadata, legacyDeleteHorizonMs, upperBoundOffset);
        }
        stats.bufferUtilization = this.offsetMap.utilization();
        stats.allDone();
        return Map.entry(endOffset, stats);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cleanSegments(UnifiedLog log, List<LogSegment> segments, OffsetMap map, long currentTime, CleanerStats stats, CleanedTransactionMetadata transactionMetadata, long legacyDeleteHorizonMs, long upperBoundOffsetOfCleaningRound) throws IOException {
        LogSegment cleaned = UnifiedLog.createNewCleanedSegment(log.dir(), log.config(), segments.get(0).baseOffset());
        transactionMetadata.setCleanedIndex(Optional.of(cleaned.txnIndex()));
        try {
            Iterator<LogSegment> iter = segments.iterator();
            Optional<LogSegment> currentSegmentOpt = Optional.of(iter.next());
            Map<Long, LastRecord> lastOffsetOfActiveProducers = log.lastRecordsOfActiveProducers();
            while (currentSegmentOpt.isPresent()) {
                LogSegment currentSegment = currentSegmentOpt.get();
                Optional<LogSegment> nextSegmentOpt = iter.hasNext() ? Optional.of(iter.next()) : Optional.empty();
                long startOffset = currentSegment.baseOffset();
                long upperBoundOffset = nextSegmentOpt.map(LogSegment::baseOffset).orElse(currentSegment.readNextOffset());
                List<AbortedTxn> abortedTransactions = log.collectAbortedTransactions(startOffset, upperBoundOffset);
                transactionMetadata.addAbortedTransactions(abortedTransactions);
                boolean retainLegacyDeletesAndTxnMarkers = currentSegment.lastModified() > legacyDeleteHorizonMs;
                this.logger.info("Cleaning {} in log {} into {} with an upper bound deletion horizon {} computed from the segment last modified time of {},{} deletes.", new Object[]{currentSegment, log.name(), cleaned.baseOffset(), legacyDeleteHorizonMs, currentSegment.lastModified(), retainLegacyDeletesAndTxnMarkers ? "retaining" : "discarding"});
                try {
                    this.cleanInto(log.topicPartition(), currentSegment.log(), cleaned, map, retainLegacyDeletesAndTxnMarkers, log.config().deleteRetentionMs, log.config().maxMessageSize(), transactionMetadata, lastOffsetOfActiveProducers, upperBoundOffsetOfCleaningRound, stats, currentTime);
                }
                catch (LogSegmentOffsetOverflowException e) {
                    this.logger.info("Caught segment overflow error during cleaning: {}", (Object)e.getMessage());
                    log.splitOverflowedSegment(currentSegment);
                    throw new LogCleaningAbortedException();
                }
                currentSegmentOpt = nextSegmentOpt;
            }
            cleaned.onBecomeInactiveSegment();
            cleaned.flush();
            long modified = segments.get(segments.size() - 1).lastModified();
            cleaned.setLastModified(modified);
            this.logger.info("Swapping in cleaned segment {} for segment(s) {} in log {}", new Object[]{cleaned, segments, log});
            log.replaceSegments(List.of(cleaned), segments);
        }
        catch (LogCleaningAbortedException e) {
            try {
                cleaned.deleteIfExists();
            }
            catch (Exception deleteException) {
                e.addSuppressed(deleteException);
            }
            finally {
                throw e;
            }
        }
    }

    private void cleanInto(TopicPartition topicPartition, FileRecords sourceRecords, LogSegment dest, final OffsetMap map, final boolean retainLegacyDeletesAndTxnMarkers, long deleteRetentionMs, int maxLogMessageSize, final CleanedTransactionMetadata transactionMetadata, final Map<Long, LastRecord> lastRecordsOfActiveProducers, final long upperBoundOffsetOfCleaningRound, final CleanerStats stats, long currentTime) throws IOException {
        MemoryRecords.RecordFilter logCleanerFilter = new MemoryRecords.RecordFilter(currentTime, deleteRetentionMs){
            private boolean discardBatchRecords;

            public MemoryRecords.RecordFilter.BatchRetentionResult checkBatchRetention(RecordBatch batch) {
                boolean canDiscardBatch = Cleaner.this.shouldDiscardBatch(batch, transactionMetadata);
                this.discardBatchRecords = batch.isControlBatch() ? canDiscardBatch && batch.deleteHorizonMs().isPresent() && batch.deleteHorizonMs().getAsLong() <= this.currentTime : canDiscardBatch;
                boolean isBatchLastRecordOfProducer = Optional.ofNullable((LastRecord)lastRecordsOfActiveProducers.get(batch.producerId())).map(lastRecord -> {
                    if (lastRecord.lastDataOffset().isPresent()) {
                        return batch.lastOffset() == lastRecord.lastDataOffset().getAsLong();
                    }
                    return batch.isControlBatch() && batch.producerEpoch() == lastRecord.producerEpoch();
                }).orElse(false);
                MemoryRecords.RecordFilter.BatchRetention batchRetention = batch.hasProducerId() && isBatchLastRecordOfProducer ? MemoryRecords.RecordFilter.BatchRetention.RETAIN_EMPTY : (batch.nextOffset() == upperBoundOffsetOfCleaningRound ? MemoryRecords.RecordFilter.BatchRetention.RETAIN_EMPTY : (this.discardBatchRecords ? MemoryRecords.RecordFilter.BatchRetention.DELETE : MemoryRecords.RecordFilter.BatchRetention.DELETE_EMPTY));
                return new MemoryRecords.RecordFilter.BatchRetentionResult(batchRetention, canDiscardBatch && batch.isControlBatch());
            }

            public boolean shouldRetainRecord(RecordBatch batch, Record record) {
                if (this.discardBatchRecords) {
                    return false;
                }
                if (batch.isControlBatch()) {
                    return true;
                }
                try {
                    return Cleaner.this.shouldRetainRecord(map, retainLegacyDeletesAndTxnMarkers, batch, record, stats, this.currentTime);
                }
                catch (DigestException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        int position = 0;
        while (position < sourceRecords.sizeInBytes()) {
            this.checkDone.accept(topicPartition);
            this.readBuffer.clear();
            this.writeBuffer.clear();
            sourceRecords.readInto(this.readBuffer, position);
            MemoryRecords records = MemoryRecords.readableRecords((ByteBuffer)this.readBuffer);
            this.throttler.maybeThrottle(records.sizeInBytes());
            MemoryRecords.FilterResult result = records.filterTo(logCleanerFilter, this.writeBuffer, this.decompressionBufferSupplier);
            stats.readMessages(result.messagesRead(), result.bytesRead());
            stats.recopyMessages(result.messagesRetained(), result.bytesRetained());
            position += result.bytesRead();
            ByteBuffer outputBuffer = result.outputBuffer();
            if (outputBuffer.position() > 0) {
                outputBuffer.flip();
                MemoryRecords retained = MemoryRecords.readableRecords((ByteBuffer)outputBuffer);
                dest.append(result.maxOffset(), retained);
                this.throttler.maybeThrottle(outputBuffer.limit());
            }
            if (this.readBuffer.limit() <= 0 || result.bytesRead() != 0) continue;
            this.growBuffersOrFail(sourceRecords, position, maxLogMessageSize, records);
        }
        this.restoreBuffers();
    }

    private void growBuffersOrFail(FileRecords sourceRecords, int position, int maxLogMessageSize, MemoryRecords memoryRecords) throws IOException {
        int maxSize;
        if (this.readBuffer.capacity() >= maxLogMessageSize) {
            Integer nextBatchSize = memoryRecords.firstBatchSize();
            String logDesc = String.format("log segment %s at position %d", sourceRecords.file(), position);
            if (nextBatchSize == null) {
                throw new IllegalStateException("Could not determine next batch size for " + logDesc);
            }
            if (nextBatchSize <= 0) {
                throw new IllegalStateException("Invalid batch size " + nextBatchSize + " for " + logDesc);
            }
            if (nextBatchSize <= this.readBuffer.capacity()) {
                throw new IllegalStateException("Batch size " + nextBatchSize + " < buffer size " + this.readBuffer.capacity() + ", but not processed for " + logDesc);
            }
            long bytesLeft = sourceRecords.channel().size() - (long)position;
            if ((long)nextBatchSize.intValue() > bytesLeft) {
                throw new CorruptRecordException("Log segment may be corrupt, batch size " + nextBatchSize + " > " + bytesLeft + " bytes left in segment for " + logDesc);
            }
            maxSize = nextBatchSize;
        } else {
            maxSize = maxLogMessageSize;
        }
        this.growBuffers(maxSize);
    }

    private boolean shouldDiscardBatch(RecordBatch batch, CleanedTransactionMetadata transactionMetadata) {
        if (batch.isControlBatch()) {
            return transactionMetadata.onControlBatchRead(batch);
        }
        return transactionMetadata.onBatchRead(batch);
    }

    private boolean shouldRetainRecord(OffsetMap map, boolean retainDeletesForLegacyRecords, RecordBatch batch, Record record, CleanerStats stats, long currentTime) throws DigestException {
        boolean pastLatestOffset;
        boolean bl = pastLatestOffset = record.offset() > map.latestOffset();
        if (pastLatestOffset) {
            return true;
        }
        if (record.hasKey()) {
            boolean legacyRecord;
            ByteBuffer key = record.key();
            long foundOffset = map.get(key);
            boolean latestOffsetForKey = record.offset() >= foundOffset;
            boolean bl2 = legacyRecord = batch.magic() < 2;
            boolean shouldRetainDeletes = !legacyRecord ? batch.deleteHorizonMs().isEmpty() || currentTime < batch.deleteHorizonMs().getAsLong() : retainDeletesForLegacyRecords;
            boolean isRetainedValue = record.hasValue() || shouldRetainDeletes;
            return latestOffsetForKey && isRetainedValue;
        }
        stats.invalidMessage();
        return false;
    }

    private void growBuffers(int maxLogMessageSize) {
        int maxBufferSize = Math.max(maxLogMessageSize, this.maxIoBufferSize);
        if (this.readBuffer.capacity() >= maxBufferSize || this.writeBuffer.capacity() >= maxBufferSize) {
            throw new IllegalStateException("This log contains a message larger than maximum allowable size of " + maxBufferSize + ".");
        }
        int newSize = Math.min(this.readBuffer.capacity() * 2, maxBufferSize);
        this.logger.info("Growing cleaner I/O buffers from {} bytes to {} bytes.", (Object)this.readBuffer.capacity(), (Object)newSize);
        this.readBuffer = ByteBuffer.allocate(newSize);
        this.writeBuffer = ByteBuffer.allocate(newSize);
    }

    private void restoreBuffers() {
        if (this.readBuffer.capacity() > this.ioBufferSize) {
            this.readBuffer = ByteBuffer.allocate(this.ioBufferSize);
        }
        if (this.writeBuffer.capacity() > this.ioBufferSize) {
            this.writeBuffer = ByteBuffer.allocate(this.ioBufferSize);
        }
    }

    public List<List<LogSegment>> groupSegmentsBySize(List<LogSegment> segments, int maxSize, int maxIndexSize, long firstUncleanableOffset) throws IOException {
        ArrayList<List<LogSegment>> grouped = new ArrayList<List<LogSegment>>();
        while (!segments.isEmpty()) {
            ArrayList<LogSegment> group = new ArrayList<LogSegment>();
            group.add(segments.get(0));
            long logSize = segments.get(0).size();
            long indexSize = segments.get(0).offsetIndex().sizeInBytes();
            long timeIndexSize = segments.get(0).timeIndex().sizeInBytes();
            segments = segments.subList(1, segments.size());
            while (!(segments.isEmpty() || logSize + (long)segments.get(0).size() > (long)maxSize || indexSize + (long)segments.get(0).offsetIndex().sizeInBytes() > (long)maxIndexSize || timeIndexSize + (long)segments.get(0).timeIndex().sizeInBytes() > (long)maxIndexSize || segments.get(0).size() != 0 && this.lastOffsetForFirstSegment(segments, firstUncleanableOffset) - ((LogSegment)group.get(group.size() - 1)).baseOffset() > Integer.MAX_VALUE)) {
                group.add(0, segments.get(0));
                logSize += (long)segments.get(0).size();
                indexSize += (long)segments.get(0).offsetIndex().sizeInBytes();
                timeIndexSize += (long)segments.get(0).timeIndex().sizeInBytes();
                segments = segments.subList(1, segments.size());
            }
            Collections.reverse(group);
            grouped.add(0, group);
        }
        Collections.reverse(grouped);
        return grouped;
    }

    private long lastOffsetForFirstSegment(List<LogSegment> segs, long firstUncleanableOffset) {
        if (segs.size() > 1) {
            return segs.get(1).baseOffset() - 1L;
        }
        return firstUncleanableOffset - 1L;
    }

    public void buildOffsetMap(UnifiedLog log, long start, long end, OffsetMap map, CleanerStats stats) throws IOException, DigestException {
        map.clear();
        List<LogSegment> dirty = log.logSegments(start, end);
        ArrayList<Long> nextSegmentStartOffsets = new ArrayList<Long>();
        if (!dirty.isEmpty()) {
            for (int i = 1; i < dirty.size(); ++i) {
                nextSegmentStartOffsets.add(dirty.get(i).baseOffset());
            }
            nextSegmentStartOffsets.add(end);
        }
        this.logger.info("Building offset map for log {} for {} segments in offset range [{}, {}).", new Object[]{log.name(), dirty.size(), start, end});
        CleanedTransactionMetadata transactionMetadata = new CleanedTransactionMetadata();
        List<AbortedTxn> abortedTransactions = log.collectAbortedTransactions(start, end);
        transactionMetadata.addAbortedTransactions(abortedTransactions);
        boolean full = false;
        for (int i = 0; i < dirty.size() && !full; ++i) {
            LogSegment segment = dirty.get(i);
            long nextSegmentStartOffset = (Long)nextSegmentStartOffsets.get(i);
            this.checkDone.accept(log.topicPartition());
            full = this.buildOffsetMapForSegment(log.topicPartition(), segment, map, start, nextSegmentStartOffset, log.config().maxMessageSize(), transactionMetadata, stats);
            if (!full) continue;
            this.logger.debug("Offset map is full, {} segments fully mapped, segment with base offset {} is partially mapped", (Object)dirty.indexOf(segment), (Object)segment.baseOffset());
        }
        this.logger.info("Offset map for log {} complete.", (Object)log.name());
    }

    private boolean buildOffsetMapForSegment(TopicPartition topicPartition, LogSegment segment, OffsetMap map, long startOffset, long nextSegmentStartOffset, int maxLogMessageSize, CleanedTransactionMetadata transactionMetadata, CleanerStats stats) throws IOException, DigestException {
        int position = segment.offsetIndex().lookup(startOffset).position();
        int maxDesiredMapSize = (int)((double)map.slots() * this.dupBufferLoadFactor);
        while (position < segment.log().sizeInBytes()) {
            this.checkDone.accept(topicPartition);
            this.readBuffer.clear();
            try {
                segment.log().readInto(this.readBuffer, position);
            }
            catch (Exception e) {
                throw new KafkaException("Failed to read from segment " + String.valueOf(segment) + " of partition " + String.valueOf(topicPartition) + " while loading offset map", (Throwable)e);
            }
            MemoryRecords records = MemoryRecords.readableRecords((ByteBuffer)this.readBuffer);
            this.throttler.maybeThrottle(records.sizeInBytes());
            int startPosition = position;
            for (MutableRecordBatch batch : records.batches()) {
                if (batch.isControlBatch()) {
                    transactionMetadata.onControlBatchRead((RecordBatch)batch);
                    stats.indexMessagesRead(1);
                } else {
                    boolean isAborted = transactionMetadata.onBatchRead((RecordBatch)batch);
                    if (isAborted) {
                        stats.indexMessagesRead(batch.countOrNull());
                    } else {
                        try (CloseableIterator recordsIterator = batch.streamingIterator(this.decompressionBufferSupplier);){
                            for (Record record : () -> recordsIterator) {
                                if (record.hasKey() && record.offset() >= startOffset) {
                                    if (map.size() < maxDesiredMapSize) {
                                        map.put(record.key(), record.offset());
                                    } else {
                                        boolean bl = true;
                                        return bl;
                                    }
                                }
                                stats.indexMessagesRead(1);
                            }
                        }
                    }
                }
                if (batch.lastOffset() < startOffset) continue;
                map.updateLatestOffset(batch.lastOffset());
            }
            int bytesRead = records.validBytes();
            stats.indexBytesRead(bytesRead);
            if ((position += bytesRead) != startPosition) continue;
            this.growBuffersOrFail(segment.log(), position, maxLogMessageSize, records);
        }
        map.updateLatestOffset(nextSegmentStartOffset - 1L);
        this.restoreBuffers();
        return false;
    }
}

