﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iotsitewise/IoTSiteWiseRequest.h>
#include <aws/iotsitewise/IoTSiteWise_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IoTSiteWise {
namespace Model {

/**
 */
class DescribeAssetPropertyRequest : public IoTSiteWiseRequest {
 public:
  AWS_IOTSITEWISE_API DescribeAssetPropertyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeAssetProperty"; }

  AWS_IOTSITEWISE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the asset. This can be either the actual ID in UUID format, or else
   * <code>externalId:</code> followed by the external ID, if it has one. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-id-references">Referencing
   * objects with external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetAssetId() const { return m_assetId; }
  inline bool AssetIdHasBeenSet() const { return m_assetIdHasBeenSet; }
  template <typename AssetIdT = Aws::String>
  void SetAssetId(AssetIdT&& value) {
    m_assetIdHasBeenSet = true;
    m_assetId = std::forward<AssetIdT>(value);
  }
  template <typename AssetIdT = Aws::String>
  DescribeAssetPropertyRequest& WithAssetId(AssetIdT&& value) {
    SetAssetId(std::forward<AssetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the asset property. This can be either the actual ID in UUID
   * format, or else <code>externalId:</code> followed by the external ID, if it has
   * one. For more information, see <a
   * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/object-ids.html#external-id-references">Referencing
   * objects with external IDs</a> in the <i>IoT SiteWise User Guide</i>.</p>
   */
  inline const Aws::String& GetPropertyId() const { return m_propertyId; }
  inline bool PropertyIdHasBeenSet() const { return m_propertyIdHasBeenSet; }
  template <typename PropertyIdT = Aws::String>
  void SetPropertyId(PropertyIdT&& value) {
    m_propertyIdHasBeenSet = true;
    m_propertyId = std::forward<PropertyIdT>(value);
  }
  template <typename PropertyIdT = Aws::String>
  DescribeAssetPropertyRequest& WithPropertyId(PropertyIdT&& value) {
    SetPropertyId(std::forward<PropertyIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_assetId;
  bool m_assetIdHasBeenSet = false;

  Aws::String m_propertyId;
  bool m_propertyIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTSiteWise
}  // namespace Aws
