﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/autoscaling/AutoScaling_EXPORTS.h>
#include <aws/autoscaling/model/ScalingActivityStatusCode.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace AutoScaling {
namespace Model {

/**
 * <p>Describes scaling activity, which is a long-running process that represents a
 * change to your Auto Scaling group, such as changing its size or replacing an
 * instance.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/Activity">AWS
 * API Reference</a></p>
 */
class Activity {
 public:
  AWS_AUTOSCALING_API Activity() = default;
  AWS_AUTOSCALING_API Activity(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_AUTOSCALING_API Activity& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_AUTOSCALING_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_AUTOSCALING_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The ID of the activity.</p>
   */
  inline const Aws::String& GetActivityId() const { return m_activityId; }
  inline bool ActivityIdHasBeenSet() const { return m_activityIdHasBeenSet; }
  template <typename ActivityIdT = Aws::String>
  void SetActivityId(ActivityIdT&& value) {
    m_activityIdHasBeenSet = true;
    m_activityId = std::forward<ActivityIdT>(value);
  }
  template <typename ActivityIdT = Aws::String>
  Activity& WithActivityId(ActivityIdT&& value) {
    SetActivityId(std::forward<ActivityIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the Auto Scaling group.</p>
   */
  inline const Aws::String& GetAutoScalingGroupName() const { return m_autoScalingGroupName; }
  inline bool AutoScalingGroupNameHasBeenSet() const { return m_autoScalingGroupNameHasBeenSet; }
  template <typename AutoScalingGroupNameT = Aws::String>
  void SetAutoScalingGroupName(AutoScalingGroupNameT&& value) {
    m_autoScalingGroupNameHasBeenSet = true;
    m_autoScalingGroupName = std::forward<AutoScalingGroupNameT>(value);
  }
  template <typename AutoScalingGroupNameT = Aws::String>
  Activity& WithAutoScalingGroupName(AutoScalingGroupNameT&& value) {
    SetAutoScalingGroupName(std::forward<AutoScalingGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A friendly, more verbose description of the activity.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Activity& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The reason the activity began.</p>
   */
  inline const Aws::String& GetCause() const { return m_cause; }
  inline bool CauseHasBeenSet() const { return m_causeHasBeenSet; }
  template <typename CauseT = Aws::String>
  void SetCause(CauseT&& value) {
    m_causeHasBeenSet = true;
    m_cause = std::forward<CauseT>(value);
  }
  template <typename CauseT = Aws::String>
  Activity& WithCause(CauseT&& value) {
    SetCause(std::forward<CauseT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The start time of the activity.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  Activity& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end time of the activity.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  Activity& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current status of the activity.</p>
   */
  inline ScalingActivityStatusCode GetStatusCode() const { return m_statusCode; }
  inline bool StatusCodeHasBeenSet() const { return m_statusCodeHasBeenSet; }
  inline void SetStatusCode(ScalingActivityStatusCode value) {
    m_statusCodeHasBeenSet = true;
    m_statusCode = value;
  }
  inline Activity& WithStatusCode(ScalingActivityStatusCode value) {
    SetStatusCode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A friendly, more verbose description of the activity status.</p>
   */
  inline const Aws::String& GetStatusMessage() const { return m_statusMessage; }
  inline bool StatusMessageHasBeenSet() const { return m_statusMessageHasBeenSet; }
  template <typename StatusMessageT = Aws::String>
  void SetStatusMessage(StatusMessageT&& value) {
    m_statusMessageHasBeenSet = true;
    m_statusMessage = std::forward<StatusMessageT>(value);
  }
  template <typename StatusMessageT = Aws::String>
  Activity& WithStatusMessage(StatusMessageT&& value) {
    SetStatusMessage(std::forward<StatusMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value between 0 and 100 that indicates the progress of the activity.</p>
   */
  inline int GetProgress() const { return m_progress; }
  inline bool ProgressHasBeenSet() const { return m_progressHasBeenSet; }
  inline void SetProgress(int value) {
    m_progressHasBeenSet = true;
    m_progress = value;
  }
  inline Activity& WithProgress(int value) {
    SetProgress(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The details about the activity.</p>
   */
  inline const Aws::String& GetDetails() const { return m_details; }
  inline bool DetailsHasBeenSet() const { return m_detailsHasBeenSet; }
  template <typename DetailsT = Aws::String>
  void SetDetails(DetailsT&& value) {
    m_detailsHasBeenSet = true;
    m_details = std::forward<DetailsT>(value);
  }
  template <typename DetailsT = Aws::String>
  Activity& WithDetails(DetailsT&& value) {
    SetDetails(std::forward<DetailsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state of the Auto Scaling group, which is either <code>InService</code>
   * or <code>Deleted</code>.</p>
   */
  inline const Aws::String& GetAutoScalingGroupState() const { return m_autoScalingGroupState; }
  inline bool AutoScalingGroupStateHasBeenSet() const { return m_autoScalingGroupStateHasBeenSet; }
  template <typename AutoScalingGroupStateT = Aws::String>
  void SetAutoScalingGroupState(AutoScalingGroupStateT&& value) {
    m_autoScalingGroupStateHasBeenSet = true;
    m_autoScalingGroupState = std::forward<AutoScalingGroupStateT>(value);
  }
  template <typename AutoScalingGroupStateT = Aws::String>
  Activity& WithAutoScalingGroupState(AutoScalingGroupStateT&& value) {
    SetAutoScalingGroupState(std::forward<AutoScalingGroupStateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Auto Scaling group.</p>
   */
  inline const Aws::String& GetAutoScalingGroupARN() const { return m_autoScalingGroupARN; }
  inline bool AutoScalingGroupARNHasBeenSet() const { return m_autoScalingGroupARNHasBeenSet; }
  template <typename AutoScalingGroupARNT = Aws::String>
  void SetAutoScalingGroupARN(AutoScalingGroupARNT&& value) {
    m_autoScalingGroupARNHasBeenSet = true;
    m_autoScalingGroupARN = std::forward<AutoScalingGroupARNT>(value);
  }
  template <typename AutoScalingGroupARNT = Aws::String>
  Activity& WithAutoScalingGroupARN(AutoScalingGroupARNT&& value) {
    SetAutoScalingGroupARN(std::forward<AutoScalingGroupARNT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_activityId;
  bool m_activityIdHasBeenSet = false;

  Aws::String m_autoScalingGroupName;
  bool m_autoScalingGroupNameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_cause;
  bool m_causeHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  ScalingActivityStatusCode m_statusCode{ScalingActivityStatusCode::NOT_SET};
  bool m_statusCodeHasBeenSet = false;

  Aws::String m_statusMessage;
  bool m_statusMessageHasBeenSet = false;

  int m_progress{0};
  bool m_progressHasBeenSet = false;

  Aws::String m_details;
  bool m_detailsHasBeenSet = false;

  Aws::String m_autoScalingGroupState;
  bool m_autoScalingGroupStateHasBeenSet = false;

  Aws::String m_autoScalingGroupARN;
  bool m_autoScalingGroupARNHasBeenSet = false;
};

}  // namespace Model
}  // namespace AutoScaling
}  // namespace Aws
