﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeconnections/CodeConnectionsRequest.h>
#include <aws/codeconnections/CodeConnections_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeConnections {
namespace Model {

/**
 */
class UpdateRepositoryLinkRequest : public CodeConnectionsRequest {
 public:
  AWS_CODECONNECTIONS_API UpdateRepositoryLinkRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateRepositoryLink"; }

  AWS_CODECONNECTIONS_API Aws::String SerializePayload() const override;

  AWS_CODECONNECTIONS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the connection for the repository link to
   * be updated. The updated connection ARN must have the same providerType (such as
   * GitHub) as the original connection ARN for the repo link.</p>
   */
  inline const Aws::String& GetConnectionArn() const { return m_connectionArn; }
  inline bool ConnectionArnHasBeenSet() const { return m_connectionArnHasBeenSet; }
  template <typename ConnectionArnT = Aws::String>
  void SetConnectionArn(ConnectionArnT&& value) {
    m_connectionArnHasBeenSet = true;
    m_connectionArn = std::forward<ConnectionArnT>(value);
  }
  template <typename ConnectionArnT = Aws::String>
  UpdateRepositoryLinkRequest& WithConnectionArn(ConnectionArnT&& value) {
    SetConnectionArn(std::forward<ConnectionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the encryption key for the repository link
   * to be updated.</p>
   */
  inline const Aws::String& GetEncryptionKeyArn() const { return m_encryptionKeyArn; }
  inline bool EncryptionKeyArnHasBeenSet() const { return m_encryptionKeyArnHasBeenSet; }
  template <typename EncryptionKeyArnT = Aws::String>
  void SetEncryptionKeyArn(EncryptionKeyArnT&& value) {
    m_encryptionKeyArnHasBeenSet = true;
    m_encryptionKeyArn = std::forward<EncryptionKeyArnT>(value);
  }
  template <typename EncryptionKeyArnT = Aws::String>
  UpdateRepositoryLinkRequest& WithEncryptionKeyArn(EncryptionKeyArnT&& value) {
    SetEncryptionKeyArn(std::forward<EncryptionKeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the repository link to be updated.</p>
   */
  inline const Aws::String& GetRepositoryLinkId() const { return m_repositoryLinkId; }
  inline bool RepositoryLinkIdHasBeenSet() const { return m_repositoryLinkIdHasBeenSet; }
  template <typename RepositoryLinkIdT = Aws::String>
  void SetRepositoryLinkId(RepositoryLinkIdT&& value) {
    m_repositoryLinkIdHasBeenSet = true;
    m_repositoryLinkId = std::forward<RepositoryLinkIdT>(value);
  }
  template <typename RepositoryLinkIdT = Aws::String>
  UpdateRepositoryLinkRequest& WithRepositoryLinkId(RepositoryLinkIdT&& value) {
    SetRepositoryLinkId(std::forward<RepositoryLinkIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_connectionArn;
  bool m_connectionArnHasBeenSet = false;

  Aws::String m_encryptionKeyArn;
  bool m_encryptionKeyArnHasBeenSet = false;

  Aws::String m_repositoryLinkId;
  bool m_repositoryLinkIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeConnections
}  // namespace Aws
