﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/databrew/GlueDataBrew_EXPORTS.h>
#include <aws/databrew/model/DatabaseOutputMode.h>
#include <aws/databrew/model/DatabaseTableOutputOptions.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GlueDataBrew {
namespace Model {

/**
 * <p>Represents a JDBC database output object which defines the output destination
 * for a DataBrew recipe job to write into.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/databrew-2017-07-25/DatabaseOutput">AWS
 * API Reference</a></p>
 */
class DatabaseOutput {
 public:
  AWS_GLUEDATABREW_API DatabaseOutput() = default;
  AWS_GLUEDATABREW_API DatabaseOutput(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUEDATABREW_API DatabaseOutput& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUEDATABREW_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Glue connection that stores the connection information for the target
   * database.</p>
   */
  inline const Aws::String& GetGlueConnectionName() const { return m_glueConnectionName; }
  inline bool GlueConnectionNameHasBeenSet() const { return m_glueConnectionNameHasBeenSet; }
  template <typename GlueConnectionNameT = Aws::String>
  void SetGlueConnectionName(GlueConnectionNameT&& value) {
    m_glueConnectionNameHasBeenSet = true;
    m_glueConnectionName = std::forward<GlueConnectionNameT>(value);
  }
  template <typename GlueConnectionNameT = Aws::String>
  DatabaseOutput& WithGlueConnectionName(GlueConnectionNameT&& value) {
    SetGlueConnectionName(std::forward<GlueConnectionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Represents options that specify how and where DataBrew writes the database
   * output generated by recipe jobs.</p>
   */
  inline const DatabaseTableOutputOptions& GetDatabaseOptions() const { return m_databaseOptions; }
  inline bool DatabaseOptionsHasBeenSet() const { return m_databaseOptionsHasBeenSet; }
  template <typename DatabaseOptionsT = DatabaseTableOutputOptions>
  void SetDatabaseOptions(DatabaseOptionsT&& value) {
    m_databaseOptionsHasBeenSet = true;
    m_databaseOptions = std::forward<DatabaseOptionsT>(value);
  }
  template <typename DatabaseOptionsT = DatabaseTableOutputOptions>
  DatabaseOutput& WithDatabaseOptions(DatabaseOptionsT&& value) {
    SetDatabaseOptions(std::forward<DatabaseOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The output mode to write into the database. Currently supported option:
   * NEW_TABLE.</p>
   */
  inline DatabaseOutputMode GetDatabaseOutputMode() const { return m_databaseOutputMode; }
  inline bool DatabaseOutputModeHasBeenSet() const { return m_databaseOutputModeHasBeenSet; }
  inline void SetDatabaseOutputMode(DatabaseOutputMode value) {
    m_databaseOutputModeHasBeenSet = true;
    m_databaseOutputMode = value;
  }
  inline DatabaseOutput& WithDatabaseOutputMode(DatabaseOutputMode value) {
    SetDatabaseOutputMode(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_glueConnectionName;
  bool m_glueConnectionNameHasBeenSet = false;

  DatabaseTableOutputOptions m_databaseOptions;
  bool m_databaseOptionsHasBeenSet = false;

  DatabaseOutputMode m_databaseOutputMode{DatabaseOutputMode::NOT_SET};
  bool m_databaseOutputModeHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlueDataBrew
}  // namespace Aws
