﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/dynamodb/DynamoDBRequest.h>
#include <aws/dynamodb/DynamoDB_EXPORTS.h>
#include <aws/dynamodb/model/BatchStatementRequest.h>
#include <aws/dynamodb/model/ReturnConsumedCapacity.h>

#include <utility>

namespace Aws {
namespace DynamoDB {
namespace Model {

/**
 */
class BatchExecuteStatementRequest : public DynamoDBRequest {
 public:
  AWS_DYNAMODB_API BatchExecuteStatementRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "BatchExecuteStatement"; }

  AWS_DYNAMODB_API Aws::String SerializePayload() const override;

  AWS_DYNAMODB_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The list of PartiQL statements representing the batch to run.</p>
   */
  inline const Aws::Vector<BatchStatementRequest>& GetStatements() const { return m_statements; }
  inline bool StatementsHasBeenSet() const { return m_statementsHasBeenSet; }
  template <typename StatementsT = Aws::Vector<BatchStatementRequest>>
  void SetStatements(StatementsT&& value) {
    m_statementsHasBeenSet = true;
    m_statements = std::forward<StatementsT>(value);
  }
  template <typename StatementsT = Aws::Vector<BatchStatementRequest>>
  BatchExecuteStatementRequest& WithStatements(StatementsT&& value) {
    SetStatements(std::forward<StatementsT>(value));
    return *this;
  }
  template <typename StatementsT = BatchStatementRequest>
  BatchExecuteStatementRequest& AddStatements(StatementsT&& value) {
    m_statementsHasBeenSet = true;
    m_statements.emplace_back(std::forward<StatementsT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline ReturnConsumedCapacity GetReturnConsumedCapacity() const { return m_returnConsumedCapacity; }
  inline bool ReturnConsumedCapacityHasBeenSet() const { return m_returnConsumedCapacityHasBeenSet; }
  inline void SetReturnConsumedCapacity(ReturnConsumedCapacity value) {
    m_returnConsumedCapacityHasBeenSet = true;
    m_returnConsumedCapacity = value;
  }
  inline BatchExecuteStatementRequest& WithReturnConsumedCapacity(ReturnConsumedCapacity value) {
    SetReturnConsumedCapacity(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<BatchStatementRequest> m_statements;
  bool m_statementsHasBeenSet = false;

  ReturnConsumedCapacity m_returnConsumedCapacity{ReturnConsumedCapacity::NOT_SET};
  bool m_returnConsumedCapacityHasBeenSet = false;
};

}  // namespace Model
}  // namespace DynamoDB
}  // namespace Aws
