﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/events/CloudWatchEventsRequest.h>
#include <aws/events/CloudWatchEvents_EXPORTS.h>
#include <aws/events/model/RuleState.h>
#include <aws/events/model/Tag.h>

#include <utility>

namespace Aws {
namespace CloudWatchEvents {
namespace Model {

/**
 */
class PutRuleRequest : public CloudWatchEventsRequest {
 public:
  AWS_CLOUDWATCHEVENTS_API PutRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutRule"; }

  AWS_CLOUDWATCHEVENTS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHEVENTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the rule that you are creating or updating.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  PutRuleRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The scheduling expression. For example, "cron(0 20 * * ? *)" or "rate(5
   * minutes)".</p>
   */
  inline const Aws::String& GetScheduleExpression() const { return m_scheduleExpression; }
  inline bool ScheduleExpressionHasBeenSet() const { return m_scheduleExpressionHasBeenSet; }
  template <typename ScheduleExpressionT = Aws::String>
  void SetScheduleExpression(ScheduleExpressionT&& value) {
    m_scheduleExpressionHasBeenSet = true;
    m_scheduleExpression = std::forward<ScheduleExpressionT>(value);
  }
  template <typename ScheduleExpressionT = Aws::String>
  PutRuleRequest& WithScheduleExpression(ScheduleExpressionT&& value) {
    SetScheduleExpression(std::forward<ScheduleExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The event pattern. For more information, see <a
   * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-and-event-patterns.html">Events
   * and Event Patterns</a> in the <i>Amazon EventBridge User Guide</i>.</p>
   */
  inline const Aws::String& GetEventPattern() const { return m_eventPattern; }
  inline bool EventPatternHasBeenSet() const { return m_eventPatternHasBeenSet; }
  template <typename EventPatternT = Aws::String>
  void SetEventPattern(EventPatternT&& value) {
    m_eventPatternHasBeenSet = true;
    m_eventPattern = std::forward<EventPatternT>(value);
  }
  template <typename EventPatternT = Aws::String>
  PutRuleRequest& WithEventPattern(EventPatternT&& value) {
    SetEventPattern(std::forward<EventPatternT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the rule is enabled or disabled.</p>
   */
  inline RuleState GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(RuleState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline PutRuleRequest& WithState(RuleState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the rule.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  PutRuleRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role associated with the rule.</p>
   * <p>If you're setting an event bus in another account as the target and that
   * account granted permission to your account through an organization instead of
   * directly by the account ID, you must specify a <code>RoleArn</code> with proper
   * permissions in the <code>Target</code> structure, instead of here in this
   * parameter.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  PutRuleRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of key-value pairs to associate with the rule.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  PutRuleRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  PutRuleRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the event bus to associate with this rule. If you omit
   * this, the default event bus is used.</p>
   */
  inline const Aws::String& GetEventBusName() const { return m_eventBusName; }
  inline bool EventBusNameHasBeenSet() const { return m_eventBusNameHasBeenSet; }
  template <typename EventBusNameT = Aws::String>
  void SetEventBusName(EventBusNameT&& value) {
    m_eventBusNameHasBeenSet = true;
    m_eventBusName = std::forward<EventBusNameT>(value);
  }
  template <typename EventBusNameT = Aws::String>
  PutRuleRequest& WithEventBusName(EventBusNameT&& value) {
    SetEventBusName(std::forward<EventBusNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_scheduleExpression;
  bool m_scheduleExpressionHasBeenSet = false;

  Aws::String m_eventPattern;
  bool m_eventPatternHasBeenSet = false;

  RuleState m_state{RuleState::NOT_SET};
  bool m_stateHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_eventBusName;
  bool m_eventBusNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvents
}  // namespace Aws
