﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kafka/KafkaRequest.h>
#include <aws/kafka/Kafka_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Kafka {
namespace Model {

/**
 */
class UpdateBrokerCountRequest : public KafkaRequest {
 public:
  AWS_KAFKA_API UpdateBrokerCountRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateBrokerCount"; }

  AWS_KAFKA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   *
          <p>The Amazon Resource Name (ARN) that uniquely identifies the
   * cluster.</p>

   */
  inline const Aws::String& GetClusterArn() const { return m_clusterArn; }
  inline bool ClusterArnHasBeenSet() const { return m_clusterArnHasBeenSet; }
  template <typename ClusterArnT = Aws::String>
  void SetClusterArn(ClusterArnT&& value) {
    m_clusterArnHasBeenSet = true;
    m_clusterArn = std::forward<ClusterArnT>(value);
  }
  template <typename ClusterArnT = Aws::String>
  UpdateBrokerCountRequest& WithClusterArn(ClusterArnT&& value) {
    SetClusterArn(std::forward<ClusterArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   *
          <p>The version of cluster to update from. A successful operation
   * will then generate a new version.</p>

   */
  inline const Aws::String& GetCurrentVersion() const { return m_currentVersion; }
  inline bool CurrentVersionHasBeenSet() const { return m_currentVersionHasBeenSet; }
  template <typename CurrentVersionT = Aws::String>
  void SetCurrentVersion(CurrentVersionT&& value) {
    m_currentVersionHasBeenSet = true;
    m_currentVersion = std::forward<CurrentVersionT>(value);
  }
  template <typename CurrentVersionT = Aws::String>
  UpdateBrokerCountRequest& WithCurrentVersion(CurrentVersionT&& value) {
    SetCurrentVersion(std::forward<CurrentVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   *
          <p>The number of broker nodes that you want the cluster to have
   * after this operation completes successfully.</p>

   */
  inline int GetTargetNumberOfBrokerNodes() const { return m_targetNumberOfBrokerNodes; }
  inline bool TargetNumberOfBrokerNodesHasBeenSet() const { return m_targetNumberOfBrokerNodesHasBeenSet; }
  inline void SetTargetNumberOfBrokerNodes(int value) {
    m_targetNumberOfBrokerNodesHasBeenSet = true;
    m_targetNumberOfBrokerNodes = value;
  }
  inline UpdateBrokerCountRequest& WithTargetNumberOfBrokerNodes(int value) {
    SetTargetNumberOfBrokerNodes(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterArn;
  bool m_clusterArnHasBeenSet = false;

  Aws::String m_currentVersion;
  bool m_currentVersionHasBeenSet = false;

  int m_targetNumberOfBrokerNodes{0};
  bool m_targetNumberOfBrokerNodesHasBeenSet = false;
};

}  // namespace Model
}  // namespace Kafka
}  // namespace Aws
