﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/neptune/NeptuneRequest.h>
#include <aws/neptune/Neptune_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Neptune {
namespace Model {

/**
 */
class AddRoleToDBClusterRequest : public NeptuneRequest {
 public:
  AWS_NEPTUNE_API AddRoleToDBClusterRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AddRoleToDBCluster"; }

  AWS_NEPTUNE_API Aws::String SerializePayload() const override;

 protected:
  AWS_NEPTUNE_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the DB cluster to associate the IAM role with.</p>
   */
  inline const Aws::String& GetDBClusterIdentifier() const { return m_dBClusterIdentifier; }
  inline bool DBClusterIdentifierHasBeenSet() const { return m_dBClusterIdentifierHasBeenSet; }
  template <typename DBClusterIdentifierT = Aws::String>
  void SetDBClusterIdentifier(DBClusterIdentifierT&& value) {
    m_dBClusterIdentifierHasBeenSet = true;
    m_dBClusterIdentifier = std::forward<DBClusterIdentifierT>(value);
  }
  template <typename DBClusterIdentifierT = Aws::String>
  AddRoleToDBClusterRequest& WithDBClusterIdentifier(DBClusterIdentifierT&& value) {
    SetDBClusterIdentifier(std::forward<DBClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role to associate with the Neptune
   * DB cluster, for example
   * <code>arn:aws:iam::123456789012:role/NeptuneAccessRole</code>.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  AddRoleToDBClusterRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the feature for the Neptune DB cluster that the IAM role is to be
   * associated with. For the list of supported feature names, see
   * <a>DBEngineVersion</a>.</p>
   */
  inline const Aws::String& GetFeatureName() const { return m_featureName; }
  inline bool FeatureNameHasBeenSet() const { return m_featureNameHasBeenSet; }
  template <typename FeatureNameT = Aws::String>
  void SetFeatureName(FeatureNameT&& value) {
    m_featureNameHasBeenSet = true;
    m_featureName = std::forward<FeatureNameT>(value);
  }
  template <typename FeatureNameT = Aws::String>
  AddRoleToDBClusterRequest& WithFeatureName(FeatureNameT&& value) {
    SetFeatureName(std::forward<FeatureNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBClusterIdentifier;
  bool m_dBClusterIdentifierHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::String m_featureName;
  bool m_featureNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace Neptune
}  // namespace Aws
