﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/networkmanager/NetworkManagerRequest.h>
#include <aws/networkmanager/NetworkManager_EXPORTS.h>
#include <aws/networkmanager/model/Tag.h>
#include <aws/networkmanager/model/VpcOptions.h>

#include <utility>

namespace Aws {
namespace NetworkManager {
namespace Model {

/**
 */
class CreateVpcAttachmentRequest : public NetworkManagerRequest {
 public:
  AWS_NETWORKMANAGER_API CreateVpcAttachmentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateVpcAttachment"; }

  AWS_NETWORKMANAGER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of a core network for the VPC attachment.</p>
   */
  inline const Aws::String& GetCoreNetworkId() const { return m_coreNetworkId; }
  inline bool CoreNetworkIdHasBeenSet() const { return m_coreNetworkIdHasBeenSet; }
  template <typename CoreNetworkIdT = Aws::String>
  void SetCoreNetworkId(CoreNetworkIdT&& value) {
    m_coreNetworkIdHasBeenSet = true;
    m_coreNetworkId = std::forward<CoreNetworkIdT>(value);
  }
  template <typename CoreNetworkIdT = Aws::String>
  CreateVpcAttachmentRequest& WithCoreNetworkId(CoreNetworkIdT&& value) {
    SetCoreNetworkId(std::forward<CoreNetworkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the VPC.</p>
   */
  inline const Aws::String& GetVpcArn() const { return m_vpcArn; }
  inline bool VpcArnHasBeenSet() const { return m_vpcArnHasBeenSet; }
  template <typename VpcArnT = Aws::String>
  void SetVpcArn(VpcArnT&& value) {
    m_vpcArnHasBeenSet = true;
    m_vpcArn = std::forward<VpcArnT>(value);
  }
  template <typename VpcArnT = Aws::String>
  CreateVpcAttachmentRequest& WithVpcArn(VpcArnT&& value) {
    SetVpcArn(std::forward<VpcArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The subnet ARN of the VPC attachment.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSubnetArns() const { return m_subnetArns; }
  inline bool SubnetArnsHasBeenSet() const { return m_subnetArnsHasBeenSet; }
  template <typename SubnetArnsT = Aws::Vector<Aws::String>>
  void SetSubnetArns(SubnetArnsT&& value) {
    m_subnetArnsHasBeenSet = true;
    m_subnetArns = std::forward<SubnetArnsT>(value);
  }
  template <typename SubnetArnsT = Aws::Vector<Aws::String>>
  CreateVpcAttachmentRequest& WithSubnetArns(SubnetArnsT&& value) {
    SetSubnetArns(std::forward<SubnetArnsT>(value));
    return *this;
  }
  template <typename SubnetArnsT = Aws::String>
  CreateVpcAttachmentRequest& AddSubnetArns(SubnetArnsT&& value) {
    m_subnetArnsHasBeenSet = true;
    m_subnetArns.emplace_back(std::forward<SubnetArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Options for the VPC attachment.</p>
   */
  inline const VpcOptions& GetOptions() const { return m_options; }
  inline bool OptionsHasBeenSet() const { return m_optionsHasBeenSet; }
  template <typename OptionsT = VpcOptions>
  void SetOptions(OptionsT&& value) {
    m_optionsHasBeenSet = true;
    m_options = std::forward<OptionsT>(value);
  }
  template <typename OptionsT = VpcOptions>
  CreateVpcAttachmentRequest& WithOptions(OptionsT&& value) {
    SetOptions(std::forward<OptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The routing policy label to apply to the VPC attachment for traffic routing
   * decisions.</p>
   */
  inline const Aws::String& GetRoutingPolicyLabel() const { return m_routingPolicyLabel; }
  inline bool RoutingPolicyLabelHasBeenSet() const { return m_routingPolicyLabelHasBeenSet; }
  template <typename RoutingPolicyLabelT = Aws::String>
  void SetRoutingPolicyLabel(RoutingPolicyLabelT&& value) {
    m_routingPolicyLabelHasBeenSet = true;
    m_routingPolicyLabel = std::forward<RoutingPolicyLabelT>(value);
  }
  template <typename RoutingPolicyLabelT = Aws::String>
  CreateVpcAttachmentRequest& WithRoutingPolicyLabel(RoutingPolicyLabelT&& value) {
    SetRoutingPolicyLabel(std::forward<RoutingPolicyLabelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key-value tags associated with the request.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateVpcAttachmentRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateVpcAttachmentRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The client token associated with the request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateVpcAttachmentRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_coreNetworkId;
  bool m_coreNetworkIdHasBeenSet = false;

  Aws::String m_vpcArn;
  bool m_vpcArnHasBeenSet = false;

  Aws::Vector<Aws::String> m_subnetArns;
  bool m_subnetArnsHasBeenSet = false;

  VpcOptions m_options;
  bool m_optionsHasBeenSet = false;

  Aws::String m_routingPolicyLabel;
  bool m_routingPolicyLabelHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace NetworkManager
}  // namespace Aws
