﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 */
class DescribeDBProxyTargetsRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeDBProxyTargetsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeDBProxyTargets"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The identifier of the <code>DBProxyTarget</code> to describe.</p>
   */
  inline const Aws::String& GetDBProxyName() const { return m_dBProxyName; }
  inline bool DBProxyNameHasBeenSet() const { return m_dBProxyNameHasBeenSet; }
  template <typename DBProxyNameT = Aws::String>
  void SetDBProxyName(DBProxyNameT&& value) {
    m_dBProxyNameHasBeenSet = true;
    m_dBProxyName = std::forward<DBProxyNameT>(value);
  }
  template <typename DBProxyNameT = Aws::String>
  DescribeDBProxyTargetsRequest& WithDBProxyName(DBProxyNameT&& value) {
    SetDBProxyName(std::forward<DBProxyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the <code>DBProxyTargetGroup</code> to describe.</p>
   */
  inline const Aws::String& GetTargetGroupName() const { return m_targetGroupName; }
  inline bool TargetGroupNameHasBeenSet() const { return m_targetGroupNameHasBeenSet; }
  template <typename TargetGroupNameT = Aws::String>
  void SetTargetGroupName(TargetGroupNameT&& value) {
    m_targetGroupNameHasBeenSet = true;
    m_targetGroupName = std::forward<TargetGroupNameT>(value);
  }
  template <typename TargetGroupNameT = Aws::String>
  DescribeDBProxyTargetsRequest& WithTargetGroupName(TargetGroupNameT&& value) {
    SetTargetGroupName(std::forward<TargetGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This parameter is not currently supported.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeDBProxyTargetsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeDBProxyTargetsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous request. If this
   * parameter is specified, the response includes only records beyond the marker, up
   * to the value specified by <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeDBProxyTargetsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so that the remaining results can be
   * retrieved.</p> <p>Default: 100</p> <p>Constraints: Minimum 20, maximum 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeDBProxyTargetsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBProxyName;
  bool m_dBProxyNameHasBeenSet = false;

  Aws::String m_targetGroupName;
  bool m_targetGroupNameHasBeenSet = false;

  Aws::Vector<Filter> m_filters;
  bool m_filtersHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;

  int m_maxRecords{0};
  bool m_maxRecordsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
