﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rekognition/RekognitionRequest.h>
#include <aws/rekognition/Rekognition_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Rekognition {
namespace Model {

/**
 */
class DescribeProjectVersionsRequest : public RekognitionRequest {
 public:
  AWS_REKOGNITION_API DescribeProjectVersionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeProjectVersions"; }

  AWS_REKOGNITION_API Aws::String SerializePayload() const override;

  AWS_REKOGNITION_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the project that contains the model/adapter
   * you want to describe.</p>
   */
  inline const Aws::String& GetProjectArn() const { return m_projectArn; }
  inline bool ProjectArnHasBeenSet() const { return m_projectArnHasBeenSet; }
  template <typename ProjectArnT = Aws::String>
  void SetProjectArn(ProjectArnT&& value) {
    m_projectArnHasBeenSet = true;
    m_projectArn = std::forward<ProjectArnT>(value);
  }
  template <typename ProjectArnT = Aws::String>
  DescribeProjectVersionsRequest& WithProjectArn(ProjectArnT&& value) {
    SetProjectArn(std::forward<ProjectArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of model or project version names that you want to describe. You can
   * add up to 10 model or project version names to the list. If you don't specify a
   * value, all project version descriptions are returned. A version name is part of
   * a project version ARN. For example, <code>my-model.2020-01-21T09.10.15</code> is
   * the version name in the following ARN.
   * <code>arn:aws:rekognition:us-east-1:123456789012:project/getting-started/version/<i>my-model.2020-01-21T09.10.15</i>/1234567890123</code>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetVersionNames() const { return m_versionNames; }
  inline bool VersionNamesHasBeenSet() const { return m_versionNamesHasBeenSet; }
  template <typename VersionNamesT = Aws::Vector<Aws::String>>
  void SetVersionNames(VersionNamesT&& value) {
    m_versionNamesHasBeenSet = true;
    m_versionNames = std::forward<VersionNamesT>(value);
  }
  template <typename VersionNamesT = Aws::Vector<Aws::String>>
  DescribeProjectVersionsRequest& WithVersionNames(VersionNamesT&& value) {
    SetVersionNames(std::forward<VersionNamesT>(value));
    return *this;
  }
  template <typename VersionNamesT = Aws::String>
  DescribeProjectVersionsRequest& AddVersionNames(VersionNamesT&& value) {
    m_versionNamesHasBeenSet = true;
    m_versionNames.emplace_back(std::forward<VersionNamesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the previous response was incomplete (because there is more results to
   * retrieve), Amazon Rekognition returns a pagination token in the response. You
   * can use this pagination token to retrieve the next set of results. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeProjectVersionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return per paginated call. The largest value
   * you can specify is 100. If you specify a value greater than 100, a
   * ValidationException error occurs. The default value is 100. </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeProjectVersionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_projectArn;
  bool m_projectArnHasBeenSet = false;

  Aws::Vector<Aws::String> m_versionNames;
  bool m_versionNamesHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Rekognition
}  // namespace Aws
