﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/servicecatalog-appregistry/AppRegistryRequest.h>
#include <aws/servicecatalog-appregistry/AppRegistry_EXPORTS.h>
#include <aws/servicecatalog-appregistry/model/AssociationOption.h>
#include <aws/servicecatalog-appregistry/model/ResourceType.h>

#include <utility>

namespace Aws {
namespace AppRegistry {
namespace Model {

/**
 */
class AssociateResourceRequest : public AppRegistryRequest {
 public:
  AWS_APPREGISTRY_API AssociateResourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateResource"; }

  AWS_APPREGISTRY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The name, ID, or ARN of the application. </p>
   */
  inline const Aws::String& GetApplication() const { return m_application; }
  inline bool ApplicationHasBeenSet() const { return m_applicationHasBeenSet; }
  template <typename ApplicationT = Aws::String>
  void SetApplication(ApplicationT&& value) {
    m_applicationHasBeenSet = true;
    m_application = std::forward<ApplicationT>(value);
  }
  template <typename ApplicationT = Aws::String>
  AssociateResourceRequest& WithApplication(ApplicationT&& value) {
    SetApplication(std::forward<ApplicationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of resource of which the application will be associated.</p>
   */
  inline ResourceType GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  inline void SetResourceType(ResourceType value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = value;
  }
  inline AssociateResourceRequest& WithResourceType(ResourceType value) {
    SetResourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ID of the resource of which the application will be
   * associated.</p>
   */
  inline const Aws::String& GetResource() const { return m_resource; }
  inline bool ResourceHasBeenSet() const { return m_resourceHasBeenSet; }
  template <typename ResourceT = Aws::String>
  void SetResource(ResourceT&& value) {
    m_resourceHasBeenSet = true;
    m_resource = std::forward<ResourceT>(value);
  }
  template <typename ResourceT = Aws::String>
  AssociateResourceRequest& WithResource(ResourceT&& value) {
    SetResource(std::forward<ResourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Determines whether an application tag is applied or skipped. </p>
   */
  inline const Aws::Vector<AssociationOption>& GetOptions() const { return m_options; }
  inline bool OptionsHasBeenSet() const { return m_optionsHasBeenSet; }
  template <typename OptionsT = Aws::Vector<AssociationOption>>
  void SetOptions(OptionsT&& value) {
    m_optionsHasBeenSet = true;
    m_options = std::forward<OptionsT>(value);
  }
  template <typename OptionsT = Aws::Vector<AssociationOption>>
  AssociateResourceRequest& WithOptions(OptionsT&& value) {
    SetOptions(std::forward<OptionsT>(value));
    return *this;
  }
  inline AssociateResourceRequest& AddOptions(AssociationOption value) {
    m_optionsHasBeenSet = true;
    m_options.push_back(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_application;
  bool m_applicationHasBeenSet = false;

  ResourceType m_resourceType{ResourceType::NOT_SET};
  bool m_resourceTypeHasBeenSet = false;

  Aws::String m_resource;
  bool m_resourceHasBeenSet = false;

  Aws::Vector<AssociationOption> m_options;
  bool m_optionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace AppRegistry
}  // namespace Aws
