﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/dynamodb/DynamoDB_EXPORTS.h>
#include <aws/dynamodb/model/AttributeValue.h>
#include <aws/dynamodb/model/BatchStatementErrorCodeEnum.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DynamoDB {
namespace Model {

/**
 * <p> An error associated with a statement in a PartiQL batch that was run.
 * </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/BatchStatementError">AWS
 * API Reference</a></p>
 */
class BatchStatementError {
 public:
  AWS_DYNAMODB_API BatchStatementError() = default;
  AWS_DYNAMODB_API BatchStatementError(Aws::Utils::Json::JsonView jsonValue);
  AWS_DYNAMODB_API BatchStatementError& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DYNAMODB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p> The error code associated with the failed PartiQL batch statement. </p>
   */
  inline BatchStatementErrorCodeEnum GetCode() const { return m_code; }
  inline bool CodeHasBeenSet() const { return m_codeHasBeenSet; }
  inline void SetCode(BatchStatementErrorCodeEnum value) {
    m_codeHasBeenSet = true;
    m_code = value;
  }
  inline BatchStatementError& WithCode(BatchStatementErrorCodeEnum value) {
    SetCode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The error message associated with the PartiQL batch response. </p>
   */
  inline const Aws::String& GetMessage() const { return m_message; }
  inline bool MessageHasBeenSet() const { return m_messageHasBeenSet; }
  template <typename MessageT = Aws::String>
  void SetMessage(MessageT&& value) {
    m_messageHasBeenSet = true;
    m_message = std::forward<MessageT>(value);
  }
  template <typename MessageT = Aws::String>
  BatchStatementError& WithMessage(MessageT&& value) {
    SetMessage(std::forward<MessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The item which caused the condition check to fail. This will be set if
   * ReturnValuesOnConditionCheckFailure is specified as <code>ALL_OLD</code>.</p>
   */
  inline const Aws::Map<Aws::String, AttributeValue>& GetItem() const { return m_item; }
  inline bool ItemHasBeenSet() const { return m_itemHasBeenSet; }
  template <typename ItemT = Aws::Map<Aws::String, AttributeValue>>
  void SetItem(ItemT&& value) {
    m_itemHasBeenSet = true;
    m_item = std::forward<ItemT>(value);
  }
  template <typename ItemT = Aws::Map<Aws::String, AttributeValue>>
  BatchStatementError& WithItem(ItemT&& value) {
    SetItem(std::forward<ItemT>(value));
    return *this;
  }
  template <typename ItemKeyT = Aws::String, typename ItemValueT = AttributeValue>
  BatchStatementError& AddItem(ItemKeyT&& key, ItemValueT&& value) {
    m_itemHasBeenSet = true;
    m_item.emplace(std::forward<ItemKeyT>(key), std::forward<ItemValueT>(value));
    return *this;
  }
  ///@}
 private:
  BatchStatementErrorCodeEnum m_code{BatchStatementErrorCodeEnum::NOT_SET};

  Aws::String m_message;

  Aws::Map<Aws::String, AttributeValue> m_item;
  bool m_codeHasBeenSet = false;
  bool m_messageHasBeenSet = false;
  bool m_itemHasBeenSet = false;
};

}  // namespace Model
}  // namespace DynamoDB
}  // namespace Aws
