﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/StorageDescriptor.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Glue {
namespace Model {

/**
 * <p>Represents a slice of table data.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/glue-2017-03-31/Partition">AWS API
 * Reference</a></p>
 */
class Partition {
 public:
  AWS_GLUE_API Partition() = default;
  AWS_GLUE_API Partition(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUE_API Partition& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLUE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The values of the partition.</p>
   */
  inline const Aws::Vector<Aws::String>& GetValues() const { return m_values; }
  inline bool ValuesHasBeenSet() const { return m_valuesHasBeenSet; }
  template <typename ValuesT = Aws::Vector<Aws::String>>
  void SetValues(ValuesT&& value) {
    m_valuesHasBeenSet = true;
    m_values = std::forward<ValuesT>(value);
  }
  template <typename ValuesT = Aws::Vector<Aws::String>>
  Partition& WithValues(ValuesT&& value) {
    SetValues(std::forward<ValuesT>(value));
    return *this;
  }
  template <typename ValuesT = Aws::String>
  Partition& AddValues(ValuesT&& value) {
    m_valuesHasBeenSet = true;
    m_values.emplace_back(std::forward<ValuesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the catalog database in which to create the partition.</p>
   */
  inline const Aws::String& GetDatabaseName() const { return m_databaseName; }
  inline bool DatabaseNameHasBeenSet() const { return m_databaseNameHasBeenSet; }
  template <typename DatabaseNameT = Aws::String>
  void SetDatabaseName(DatabaseNameT&& value) {
    m_databaseNameHasBeenSet = true;
    m_databaseName = std::forward<DatabaseNameT>(value);
  }
  template <typename DatabaseNameT = Aws::String>
  Partition& WithDatabaseName(DatabaseNameT&& value) {
    SetDatabaseName(std::forward<DatabaseNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the database table in which to create the partition.</p>
   */
  inline const Aws::String& GetTableName() const { return m_tableName; }
  inline bool TableNameHasBeenSet() const { return m_tableNameHasBeenSet; }
  template <typename TableNameT = Aws::String>
  void SetTableName(TableNameT&& value) {
    m_tableNameHasBeenSet = true;
    m_tableName = std::forward<TableNameT>(value);
  }
  template <typename TableNameT = Aws::String>
  Partition& WithTableName(TableNameT&& value) {
    SetTableName(std::forward<TableNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time at which the partition was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  Partition& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last time at which the partition was accessed.</p>
   */
  inline const Aws::Utils::DateTime& GetLastAccessTime() const { return m_lastAccessTime; }
  inline bool LastAccessTimeHasBeenSet() const { return m_lastAccessTimeHasBeenSet; }
  template <typename LastAccessTimeT = Aws::Utils::DateTime>
  void SetLastAccessTime(LastAccessTimeT&& value) {
    m_lastAccessTimeHasBeenSet = true;
    m_lastAccessTime = std::forward<LastAccessTimeT>(value);
  }
  template <typename LastAccessTimeT = Aws::Utils::DateTime>
  Partition& WithLastAccessTime(LastAccessTimeT&& value) {
    SetLastAccessTime(std::forward<LastAccessTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Provides information about the physical location where the partition is
   * stored.</p>
   */
  inline const StorageDescriptor& GetStorageDescriptor() const { return m_storageDescriptor; }
  inline bool StorageDescriptorHasBeenSet() const { return m_storageDescriptorHasBeenSet; }
  template <typename StorageDescriptorT = StorageDescriptor>
  void SetStorageDescriptor(StorageDescriptorT&& value) {
    m_storageDescriptorHasBeenSet = true;
    m_storageDescriptor = std::forward<StorageDescriptorT>(value);
  }
  template <typename StorageDescriptorT = StorageDescriptor>
  Partition& WithStorageDescriptor(StorageDescriptorT&& value) {
    SetStorageDescriptor(std::forward<StorageDescriptorT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>These key-value pairs define partition parameters.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Map<Aws::String, Aws::String>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Map<Aws::String, Aws::String>>
  Partition& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersKeyT = Aws::String, typename ParametersValueT = Aws::String>
  Partition& AddParameters(ParametersKeyT&& key, ParametersValueT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace(std::forward<ParametersKeyT>(key), std::forward<ParametersValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last time at which column statistics were computed for this
   * partition.</p>
   */
  inline const Aws::Utils::DateTime& GetLastAnalyzedTime() const { return m_lastAnalyzedTime; }
  inline bool LastAnalyzedTimeHasBeenSet() const { return m_lastAnalyzedTimeHasBeenSet; }
  template <typename LastAnalyzedTimeT = Aws::Utils::DateTime>
  void SetLastAnalyzedTime(LastAnalyzedTimeT&& value) {
    m_lastAnalyzedTimeHasBeenSet = true;
    m_lastAnalyzedTime = std::forward<LastAnalyzedTimeT>(value);
  }
  template <typename LastAnalyzedTimeT = Aws::Utils::DateTime>
  Partition& WithLastAnalyzedTime(LastAnalyzedTimeT&& value) {
    SetLastAnalyzedTime(std::forward<LastAnalyzedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Data Catalog in which the partition resides.</p>
   */
  inline const Aws::String& GetCatalogId() const { return m_catalogId; }
  inline bool CatalogIdHasBeenSet() const { return m_catalogIdHasBeenSet; }
  template <typename CatalogIdT = Aws::String>
  void SetCatalogId(CatalogIdT&& value) {
    m_catalogIdHasBeenSet = true;
    m_catalogId = std::forward<CatalogIdT>(value);
  }
  template <typename CatalogIdT = Aws::String>
  Partition& WithCatalogId(CatalogIdT&& value) {
    SetCatalogId(std::forward<CatalogIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_values;

  Aws::String m_databaseName;

  Aws::String m_tableName;

  Aws::Utils::DateTime m_creationTime{};

  Aws::Utils::DateTime m_lastAccessTime{};

  StorageDescriptor m_storageDescriptor;

  Aws::Map<Aws::String, Aws::String> m_parameters;

  Aws::Utils::DateTime m_lastAnalyzedTime{};

  Aws::String m_catalogId;
  bool m_valuesHasBeenSet = false;
  bool m_databaseNameHasBeenSet = false;
  bool m_tableNameHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_lastAccessTimeHasBeenSet = false;
  bool m_storageDescriptorHasBeenSet = false;
  bool m_parametersHasBeenSet = false;
  bool m_lastAnalyzedTimeHasBeenSet = false;
  bool m_catalogIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
