﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DeleteNetworkAclEntryRequest : public EC2Request {
 public:
  AWS_EC2_API DeleteNetworkAclEntryRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteNetworkAclEntry"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DeleteNetworkAclEntryRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the network ACL.</p>
   */
  inline const Aws::String& GetNetworkAclId() const { return m_networkAclId; }
  inline bool NetworkAclIdHasBeenSet() const { return m_networkAclIdHasBeenSet; }
  template <typename NetworkAclIdT = Aws::String>
  void SetNetworkAclId(NetworkAclIdT&& value) {
    m_networkAclIdHasBeenSet = true;
    m_networkAclId = std::forward<NetworkAclIdT>(value);
  }
  template <typename NetworkAclIdT = Aws::String>
  DeleteNetworkAclEntryRequest& WithNetworkAclId(NetworkAclIdT&& value) {
    SetNetworkAclId(std::forward<NetworkAclIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The rule number of the entry to delete.</p>
   */
  inline int GetRuleNumber() const { return m_ruleNumber; }
  inline bool RuleNumberHasBeenSet() const { return m_ruleNumberHasBeenSet; }
  inline void SetRuleNumber(int value) {
    m_ruleNumberHasBeenSet = true;
    m_ruleNumber = value;
  }
  inline DeleteNetworkAclEntryRequest& WithRuleNumber(int value) {
    SetRuleNumber(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the rule is an egress rule.</p>
   */
  inline bool GetEgress() const { return m_egress; }
  inline bool EgressHasBeenSet() const { return m_egressHasBeenSet; }
  inline void SetEgress(bool value) {
    m_egressHasBeenSet = true;
    m_egress = value;
  }
  inline DeleteNetworkAclEntryRequest& WithEgress(bool value) {
    SetEgress(value);
    return *this;
  }
  ///@}
 private:
  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;

  Aws::String m_networkAclId;
  bool m_networkAclIdHasBeenSet = false;

  int m_ruleNumber{0};
  bool m_ruleNumberHasBeenSet = false;

  bool m_egress{false};
  bool m_egressHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
