﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/Filter.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class DescribeTrafficMirrorFiltersRequest : public EC2Request {
 public:
  AWS_EC2_API DescribeTrafficMirrorFiltersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeTrafficMirrorFilters"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of the Traffic Mirror filter.</p>
   */
  inline const Aws::Vector<Aws::String>& GetTrafficMirrorFilterIds() const { return m_trafficMirrorFilterIds; }
  inline bool TrafficMirrorFilterIdsHasBeenSet() const { return m_trafficMirrorFilterIdsHasBeenSet; }
  template <typename TrafficMirrorFilterIdsT = Aws::Vector<Aws::String>>
  void SetTrafficMirrorFilterIds(TrafficMirrorFilterIdsT&& value) {
    m_trafficMirrorFilterIdsHasBeenSet = true;
    m_trafficMirrorFilterIds = std::forward<TrafficMirrorFilterIdsT>(value);
  }
  template <typename TrafficMirrorFilterIdsT = Aws::Vector<Aws::String>>
  DescribeTrafficMirrorFiltersRequest& WithTrafficMirrorFilterIds(TrafficMirrorFilterIdsT&& value) {
    SetTrafficMirrorFilterIds(std::forward<TrafficMirrorFilterIdsT>(value));
    return *this;
  }
  template <typename TrafficMirrorFilterIdsT = Aws::String>
  DescribeTrafficMirrorFiltersRequest& AddTrafficMirrorFilterIds(TrafficMirrorFilterIdsT&& value) {
    m_trafficMirrorFilterIdsHasBeenSet = true;
    m_trafficMirrorFilterIds.emplace_back(std::forward<TrafficMirrorFilterIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline DescribeTrafficMirrorFiltersRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more filters. The possible values are:</p> <ul> <li> <p>
   * <code>description</code>: The Traffic Mirror filter description.</p> </li> <li>
   * <p> <code>traffic-mirror-filter-id</code>: The ID of the Traffic Mirror
   * filter.</p> </li> </ul>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeTrafficMirrorFiltersRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeTrafficMirrorFiltersRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return with a single call. To retrieve the
   * remaining results, make another call with the returned <code>nextToken</code>
   * value.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeTrafficMirrorFiltersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeTrafficMirrorFiltersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_trafficMirrorFilterIds;
  bool m_trafficMirrorFilterIdsHasBeenSet = false;

  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;

  Aws::Vector<Filter> m_filters;
  bool m_filtersHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
