﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class ModifyVerifiedAccessInstanceRequest : public EC2Request {
 public:
  AWS_EC2_API ModifyVerifiedAccessInstanceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ModifyVerifiedAccessInstance"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of the Verified Access instance.</p>
   */
  inline const Aws::String& GetVerifiedAccessInstanceId() const { return m_verifiedAccessInstanceId; }
  inline bool VerifiedAccessInstanceIdHasBeenSet() const { return m_verifiedAccessInstanceIdHasBeenSet; }
  template <typename VerifiedAccessInstanceIdT = Aws::String>
  void SetVerifiedAccessInstanceId(VerifiedAccessInstanceIdT&& value) {
    m_verifiedAccessInstanceIdHasBeenSet = true;
    m_verifiedAccessInstanceId = std::forward<VerifiedAccessInstanceIdT>(value);
  }
  template <typename VerifiedAccessInstanceIdT = Aws::String>
  ModifyVerifiedAccessInstanceRequest& WithVerifiedAccessInstanceId(VerifiedAccessInstanceIdT&& value) {
    SetVerifiedAccessInstanceId(std::forward<VerifiedAccessInstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description for the Verified Access instance.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  ModifyVerifiedAccessInstanceRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline ModifyVerifiedAccessInstanceRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive token that you provide to ensure idempotency of your
   * modification request. For more information, see <a
   * href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring
   * idempotency</a>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  ModifyVerifiedAccessInstanceRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The custom subdomain.</p>
   */
  inline const Aws::String& GetCidrEndpointsCustomSubDomain() const { return m_cidrEndpointsCustomSubDomain; }
  inline bool CidrEndpointsCustomSubDomainHasBeenSet() const { return m_cidrEndpointsCustomSubDomainHasBeenSet; }
  template <typename CidrEndpointsCustomSubDomainT = Aws::String>
  void SetCidrEndpointsCustomSubDomain(CidrEndpointsCustomSubDomainT&& value) {
    m_cidrEndpointsCustomSubDomainHasBeenSet = true;
    m_cidrEndpointsCustomSubDomain = std::forward<CidrEndpointsCustomSubDomainT>(value);
  }
  template <typename CidrEndpointsCustomSubDomainT = Aws::String>
  ModifyVerifiedAccessInstanceRequest& WithCidrEndpointsCustomSubDomain(CidrEndpointsCustomSubDomainT&& value) {
    SetCidrEndpointsCustomSubDomain(std::forward<CidrEndpointsCustomSubDomainT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_verifiedAccessInstanceId;
  bool m_verifiedAccessInstanceIdHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_cidrEndpointsCustomSubDomain;
  bool m_cidrEndpointsCustomSubDomainHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
