<?php

declare(strict_types=1);

namespace Drupal\Tests\content_translation\Functional;

use Drupal\Core\Url;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\Tests\language\Traits\LanguageTestTrait;

/**
 * Tests the untranslatable fields behaviors.
 *
 * @group content_translation
 */
class ContentTranslationUntranslatableFieldsTest extends ContentTranslationPendingRevisionTestBase {

  use LanguageTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['field_test'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->doSetup();

    // Configure one field as untranslatable.
    $this->drupalLogin($this->administrator);
    static::setFieldTranslatable($this->entityTypeId, $this->bundle, $this->fieldName, FALSE);

    /** @var \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager */
    $entity_field_manager = $this->container->get('entity_field.manager');
    $entity_field_manager->clearCachedFieldDefinitions();
    $definitions = $entity_field_manager->getFieldDefinitions($this->entityTypeId, $this->bundle);
    $this->assertFalse($definitions[$this->fieldName]->isTranslatable());
  }

  /**
   * {@inheritdoc}
   */
  protected function setupTestFields(): void {
    parent::setupTestFields();

    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_multilingual',
      'type' => 'test_field',
      'entity_type' => $this->entityTypeId,
      'cardinality' => 1,
    ]);
    $field_storage->save();
    FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => $this->bundle,
      'label' => 'Untranslatable-but-visible test field',
      'translatable' => FALSE,
    ])->save();
    \Drupal::service('entity_display.repository')->getFormDisplay($this->entityTypeId, $this->bundle, 'default')
      ->setComponent('field_multilingual', [
        'type' => 'test_field_widget_multilingual',
      ])
      ->save();
  }

  /**
   * Tests that hiding untranslatable field widgets works correctly.
   */
  public function testHiddenWidgets(): void {
    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager */
    $entity_type_manager = $this->container->get('entity_type.manager');
    $id = $this->createEntity(['title' => $this->randomString()], 'en');
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $entity = $entity_type_manager
      ->getStorage($this->entityTypeId)
      ->load($id);

    // Check that the untranslatable field widget is displayed on the edit form
    // and no translatability clue is displayed yet.
    $en_edit_url = $entity->toUrl('edit-form');
    $this->drupalGet($en_edit_url);
    $field_xpath = '//input[@name="' . $this->fieldName . '[0][value]"]';
    $this->assertSession()->elementExists('xpath', $field_xpath);
    $clue_xpath = '//label[@for="edit-' . strtr($this->fieldName, '_', '-') . '-0-value"]/span[text()="(all languages)"]';
    $this->assertSession()->elementNotExists('xpath', $clue_xpath);
    $this->assertSession()->pageTextContains('Untranslatable-but-visible test field');

    // Add a translation and check that the untranslatable field widget is
    // displayed on the translation and edit forms along with translatability
    // clues.
    $add_url = Url::fromRoute("entity.{$this->entityTypeId}.content_translation_add", [
      $entity->getEntityTypeId() => $entity->id(),
      'source' => 'en',
      'target' => 'it',
    ]);
    $this->drupalGet($add_url);
    $this->assertSession()->elementExists('xpath', $field_xpath);
    $this->assertSession()->elementExists('xpath', $clue_xpath);
    $this->assertSession()->pageTextContains('Untranslatable-but-visible test field');
    $this->submitForm([], 'Save');

    // Check that the widget is displayed along with its clue in the edit form
    // for both languages.
    $this->drupalGet($en_edit_url);
    $this->assertSession()->elementExists('xpath', $field_xpath);
    $this->assertSession()->elementExists('xpath', $clue_xpath);
    $it_edit_url = $entity->toUrl('edit-form', ['language' => ConfigurableLanguage::load('it')]);
    $this->drupalGet($it_edit_url);
    $this->assertSession()->elementExists('xpath', $field_xpath);
    $this->assertSession()->elementExists('xpath', $clue_xpath);

    // Configure untranslatable field widgets to be hidden on non-default
    // language edit forms.
    $settings_key = 'settings[' . $this->entityTypeId . '][' . $this->bundle . '][settings][content_translation][untranslatable_fields_hide]';
    $settings_url = 'admin/config/regional/content-language';
    $this->drupalGet($settings_url);
    $this->submitForm([$settings_key => 1], 'Save configuration');

    // Verify that the widget is displayed in the default language edit form,
    // but no clue is displayed.
    $this->drupalGet($en_edit_url);
    $field_xpath = '//input[@name="' . $this->fieldName . '[0][value]"]';
    $this->assertSession()->elementExists('xpath', $field_xpath);
    $this->assertSession()->elementNotExists('xpath', $clue_xpath);
    $this->assertSession()->pageTextContains('Untranslatable-but-visible test field');

    // Verify no widget is displayed on the non-default language edit form.
    $this->drupalGet($it_edit_url);
    $this->assertSession()->elementNotExists('xpath', $field_xpath);
    $this->assertSession()->elementNotExists('xpath', $clue_xpath);
    $this->assertSession()->pageTextContains('Untranslatable-but-visible test field');

    // Verify a warning is displayed.
    $this->assertSession()->statusMessageContains('Fields that apply to all languages are hidden to avoid conflicting changes.', 'warning');
    $this->assertSession()->elementExists('xpath', '//a[@href="' . $entity->toUrl('edit-form')->toString() . '" and text()="Edit them on the original language form"]');

    // Configure untranslatable field widgets to be displayed on non-default
    // language edit forms.
    $this->drupalGet($settings_url);
    $this->submitForm([$settings_key => 0], 'Save configuration');

    // Check that the widget is displayed along with its clue in the edit form
    // for both languages.
    $this->drupalGet($en_edit_url);
    $this->assertSession()->elementExists('xpath', $field_xpath);
    $this->assertSession()->elementExists('xpath', $clue_xpath);
    $this->drupalGet($it_edit_url);
    $this->assertSession()->elementExists('xpath', $field_xpath);
    $this->assertSession()->elementExists('xpath', $clue_xpath);

    // Enable content moderation and verify that widgets are hidden despite them
    // being configured to be displayed.
    $this->enableContentModeration();
    $this->drupalGet($it_edit_url);
    $this->assertSession()->elementNotExists('xpath', $field_xpath);
    $this->assertSession()->elementNotExists('xpath', $clue_xpath);

    // Verify a warning is displayed.
    $this->assertSession()->statusMessageContains('Fields that apply to all languages are hidden to avoid conflicting changes.', 'warning');
    $this->assertSession()->elementExists('xpath', '//a[@href="' . $entity->toUrl('edit-form')->toString() . '" and text()="Edit them on the original language form"]');

    // Verify that checkboxes on the language content settings page are checked
    // and disabled for moderated bundles.
    $this->drupalGet($settings_url);
    $field_name = "settings[{$this->entityTypeId}][{$this->bundle}][settings][content_translation][untranslatable_fields_hide]";
    $this->assertSession()->fieldValueEquals($field_name, 1);
    $this->assertSession()->fieldDisabled($field_name);
    $this->submitForm([$settings_key => 0], 'Save configuration');
    $this->assertSession()->fieldValueEquals($field_name, 1);
    $this->assertSession()->fieldDisabled($field_name);

    // Verify that the untranslatable fields warning message is not displayed
    // when submitting.
    $this->drupalGet($it_edit_url);
    $this->assertSession()->pageTextContains('Fields that apply to all languages are hidden to avoid conflicting changes.');
    $this->submitForm([], 'Save (this translation)');
    $this->assertSession()->pageTextNotContains('Fields that apply to all languages are hidden to avoid conflicting changes.');
  }

}
