// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { needs, NodeEncryptionMaterial, NodeDecryptionMaterial, unwrapDataKey, wrapWithKeyObjectIfSupported, } from '@aws-crypto/material-management';
import { createCipheriv, createDecipheriv, createSign, createVerify, timingSafeEqual, } from 'crypto';
import { HKDF } from '@aws-crypto/hkdf-node';
import { kdfInfo, kdfCommitKeyInfo } from '@aws-crypto/serialize';
const kdfIndex = Object.freeze({
    sha256: HKDF('sha256'),
    sha384: HKDF('sha384'),
    sha512: HKDF('sha512'),
});
export const getEncryptHelper = (material) => {
    /* Precondition: NodeEncryptionMaterial must have a valid data key. */
    needs(material.hasValidKey(), 'Material has no unencrypted data key.');
    const { signatureHash } = material.suite;
    /* Conditional types can not narrow the return type :(
     * Function overloads "works" but then I can not export
     * the function and have eslint be happy (Multiple exports of name)
     */
    const getCipherInfo = curryCryptoStream(material, createCipheriv);
    return Object.freeze({
        getCipherInfo,
        getSigner: signatureHash ? getSigner : undefined,
        dispose,
    });
    function getSigner() {
        /* Precondition: The NodeEncryptionMaterial must have not been zeroed.
         * hasUnencryptedDataKey will check that the unencrypted data key has been set
         * *and* that it has not been zeroed.  At this point it must have been set
         * because the KDF function operated on it.  So at this point
         * we are protecting that someone has zeroed out the material
         * because the Encrypt process has been complete.
         */
        needs(material.hasUnencryptedDataKey, 'Unencrypted data key has been zeroed.');
        if (!signatureHash)
            throw new Error('Material does not support signature.');
        const { signatureKey } = material;
        if (!signatureKey)
            throw new Error('Material does not support signature.');
        const { privateKey } = signatureKey;
        if (typeof privateKey !== 'string')
            throw new Error('Material does not support signature.');
        const signer = Object.assign(createSign(signatureHash), 
        // don't export the private key if we don't have to
        { awsCryptoSign: () => signer.sign(privateKey) });
        return signer;
    }
    function dispose() {
        material.zeroUnencryptedDataKey();
    }
};
export const getDecryptionHelper = (material) => {
    /* Precondition: NodeDecryptionMaterial must have a valid data key. */
    needs(material.hasValidKey(), 'Material has no unencrypted data key.');
    const { signatureHash } = material.suite;
    /* Conditional types can not narrow the return type :(
     * Function overloads "works" but then I can not export
     * the function and have eslint be happy (Multiple exports of name)
     */
    const getDecipherInfo = curryCryptoStream(material, createDecipheriv);
    return Object.freeze({
        getDecipherInfo,
        getVerify: signatureHash ? getVerify : undefined,
        dispose,
    });
    function getVerify() {
        if (!signatureHash)
            throw new Error('Material does not support signature.');
        const { verificationKey } = material;
        if (!verificationKey)
            throw new Error('Material does not support signature.');
        const verify = Object.assign(createVerify(signatureHash), 
        // explicitly bind the public key for this material
        {
            awsCryptoVerify: (signature) => 
            // As typescript gets better typing
            // We should consider either generics or
            // 2 different verificationKeys for Node and WebCrypto
            verify.verify(verificationKey.publicKey, signature),
        });
        return verify;
    }
    function dispose() {
        material.zeroUnencryptedDataKey();
    }
};
export function curryCryptoStream(material, createCryptoIvStream) {
    const { encryption: cipherName, ivLength } = material.suite;
    const isEncrypt = material instanceof NodeEncryptionMaterial;
    /* Precondition: material must be either NodeEncryptionMaterial or NodeDecryptionMaterial.
     *
     */
    needs(isEncrypt
        ? createCipheriv === createCryptoIvStream
        : material instanceof NodeDecryptionMaterial
            ? createDecipheriv === createCryptoIvStream
            : false, 'Unsupported cryptographic material.');
    return (messageId, commitKey) => {
        const { derivedKey, keyCommitment } = nodeKdf(material, messageId, commitKey);
        return (isEncrypt
            ? { getCipher: createCryptoStream, keyCommitment }
            : createCryptoStream);
        function createCryptoStream(iv) {
            /* Precondition: The length of the IV must match the NodeAlgorithmSuite specification. */
            needs(iv.byteLength === ivLength, 'Iv length does not match algorithm suite specification');
            /* Precondition: The material must have not been zeroed.
             * hasUnencryptedDataKey will check that the unencrypted data key has been set
             * *and* that it has not been zeroed.  At this point it must have been set
             * because the KDF function operated on it.  So at this point
             * we are protecting that someone has zeroed out the material
             * because the Encrypt process has been complete.
             */
            needs(material.hasUnencryptedDataKey, 'Unencrypted data key has been zeroed.');
            /* createDecipheriv is incorrectly typed in @types/node. It should take key: CipherKey, not key: BinaryLike.
             * Also, the check above ensures
             * that _createCryptoStream is not false.
             * But TypeScript does not believe me.
             * For any complicated code,
             * you should defer to the checker,
             * but here I'm going to assert
             * it is simple enough.
             */
            return createCryptoIvStream(cipherName, derivedKey, iv);
        }
    };
}
export function nodeKdf(material, nonce, commitKey) {
    const dataKey = material.getUnencryptedDataKey();
    const { kdf, kdfHash, keyLengthBytes, commitmentLength, saltLengthBytes, commitment, id: suiteId, } = material.suite;
    /* Check for early return (Postcondition): No Node.js KDF, just return the unencrypted data key. */
    if (!kdf && !kdfHash) {
        /* Postcondition: Non-KDF algorithm suites *must* not have a commitment. */
        needs(!commitKey, 'Commitment not supported.');
        return { derivedKey: dataKey };
    }
    /* Precondition: Valid HKDF values must exist for Node.js. */
    needs(kdf === 'HKDF' &&
        kdfHash &&
        kdfIndex[kdfHash] &&
        nonce instanceof Uint8Array, 'Invalid HKDF values.');
    /* The unwrap is done once we *know* that a KDF is required.
     * If we unwrapped before everything will work,
     * but we may be creating new copies of the unencrypted data key (export).
     */
    const { buffer: dkBuffer, byteOffset: dkByteOffset, byteLength: dkByteLength, } = unwrapDataKey(dataKey);
    if (commitment === 'NONE') {
        /* Postcondition: Non-committing Node algorithm suites *must* not have a commitment. */
        needs(!commitKey, 'Commitment not supported.');
        const toExtract = Buffer.from(dkBuffer, dkByteOffset, dkByteLength);
        const { buffer, byteOffset, byteLength } = kdfInfo(suiteId, nonce);
        const infoBuff = Buffer.from(buffer, byteOffset, byteLength);
        const derivedBytes = kdfIndex[kdfHash](toExtract)(keyLengthBytes, infoBuff);
        const derivedKey = wrapWithKeyObjectIfSupported(derivedBytes);
        return { derivedKey };
    }
    /* Precondition UNTESTED: Committing suites must define expected values. */
    needs(commitment === 'KEY' && commitmentLength && saltLengthBytes, 'Malformed suite data.');
    /* Precondition: For committing algorithms, the nonce *must* be 256 bit.
     * i.e. It must target a V2 message format.
     */
    needs(nonce.byteLength === saltLengthBytes, 'Nonce is not the correct length for committed algorithm suite.');
    const toExtract = Buffer.from(dkBuffer, dkByteOffset, dkByteLength);
    const expand = kdfIndex[kdfHash](toExtract, nonce);
    const { keyLabel, commitLabel } = kdfCommitKeyInfo(material.suite);
    const keyCommitment = expand(commitmentLength / 8, commitLabel);
    const isDecrypt = material instanceof NodeDecryptionMaterial;
    /* Precondition: If material is NodeDecryptionMaterial the key commitments *must* match.
     * This is also the preferred location to check,
     * because then the decryption key is never even derived.
     */
    needs((isDecrypt && commitKey && timingSafeEqual(keyCommitment, commitKey)) ||
        (!isDecrypt && !commitKey), isDecrypt ? 'Commitment does not match.' : 'Invalid arguments.');
    const derivedBytes = expand(keyLengthBytes, keyLabel);
    const derivedKey = wrapWithKeyObjectIfSupported(derivedBytes);
    return { derivedKey, keyCommitment };
}
//# sourceMappingURL=data:application/json;base64,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