﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/chime-sdk-meetings/ChimeSDKMeetings_EXPORTS.h>
#include <aws/chime-sdk-meetings/model/TranscribeLanguageCode.h>
#include <aws/chime-sdk-meetings/model/TranscribeVocabularyFilterMethod.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/chime-sdk-meetings/model/TranscribeRegion.h>
#include <aws/chime-sdk-meetings/model/TranscribePartialResultsStability.h>
#include <aws/chime-sdk-meetings/model/TranscribeContentIdentificationType.h>
#include <aws/chime-sdk-meetings/model/TranscribeContentRedactionType.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Json
{
  class JsonValue;
  class JsonView;
} // namespace Json
} // namespace Utils
namespace ChimeSDKMeetings
{
namespace Model
{

  /**
   * <p>Settings specific for Amazon Transcribe as the live transcription engine.
   * </p> <p>If you specify an invalid combination of parameters, a
   * <code>TranscriptFailed</code> event will be sent with the contents of the
   * <code>BadRequestException</code> generated by Amazon Transcribe. For more
   * information on each parameter and which combinations are valid, refer to the <a
   * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/API_streaming_StartStreamTranscription.html">StartStreamTranscription</a>
   * API in the <i>Amazon Transcribe Developer Guide</i>.</p><p><h3>See Also:</h3>  
   * <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/EngineTranscribeSettings">AWS
   * API Reference</a></p>
   */
  class EngineTranscribeSettings
  {
  public:
    AWS_CHIMESDKMEETINGS_API EngineTranscribeSettings() = default;
    AWS_CHIMESDKMEETINGS_API EngineTranscribeSettings(Aws::Utils::Json::JsonView jsonValue);
    AWS_CHIMESDKMEETINGS_API EngineTranscribeSettings& operator=(Aws::Utils::Json::JsonView jsonValue);
    AWS_CHIMESDKMEETINGS_API Aws::Utils::Json::JsonValue Jsonize() const;


    ///@{
    /**
     * <p>Specify the language code that represents the language spoken.</p> <p>If
     * you're unsure of the language spoken in your audio, consider using
     * <code>IdentifyLanguage</code> to enable automatic language identification.</p>
     */
    inline TranscribeLanguageCode GetLanguageCode() const { return m_languageCode; }
    inline bool LanguageCodeHasBeenSet() const { return m_languageCodeHasBeenSet; }
    inline void SetLanguageCode(TranscribeLanguageCode value) { m_languageCodeHasBeenSet = true; m_languageCode = value; }
    inline EngineTranscribeSettings& WithLanguageCode(TranscribeLanguageCode value) { SetLanguageCode(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify how you want your vocabulary filter applied to your transcript.</p>
     * <p>To replace words with <code>***</code>, choose <code>mask</code>.</p> <p>To
     * delete words, choose <code>remove</code>.</p> <p>To flag words without changing
     * them, choose <code>tag</code>.</p>
     */
    inline TranscribeVocabularyFilterMethod GetVocabularyFilterMethod() const { return m_vocabularyFilterMethod; }
    inline bool VocabularyFilterMethodHasBeenSet() const { return m_vocabularyFilterMethodHasBeenSet; }
    inline void SetVocabularyFilterMethod(TranscribeVocabularyFilterMethod value) { m_vocabularyFilterMethodHasBeenSet = true; m_vocabularyFilterMethod = value; }
    inline EngineTranscribeSettings& WithVocabularyFilterMethod(TranscribeVocabularyFilterMethod value) { SetVocabularyFilterMethod(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the name of the custom vocabulary filter that you want to use when
     * processing your transcription. Note that vocabulary filter names are case
     * sensitive. </p> <p>If you use Amazon Transcribe in multiple Regions, the
     * vocabulary filter must be available in Amazon Transcribe in each Region.</p>
     * <p>If you include <code>IdentifyLanguage</code> and want to use one or more
     * vocabulary filters with your transcription, use the
     * <code>VocabularyFilterNames</code> parameter instead.</p>
     */
    inline const Aws::String& GetVocabularyFilterName() const { return m_vocabularyFilterName; }
    inline bool VocabularyFilterNameHasBeenSet() const { return m_vocabularyFilterNameHasBeenSet; }
    template<typename VocabularyFilterNameT = Aws::String>
    void SetVocabularyFilterName(VocabularyFilterNameT&& value) { m_vocabularyFilterNameHasBeenSet = true; m_vocabularyFilterName = std::forward<VocabularyFilterNameT>(value); }
    template<typename VocabularyFilterNameT = Aws::String>
    EngineTranscribeSettings& WithVocabularyFilterName(VocabularyFilterNameT&& value) { SetVocabularyFilterName(std::forward<VocabularyFilterNameT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the name of the custom vocabulary that you want to use when
     * processing your transcription. Note that vocabulary names are case
     * sensitive.</p> <p>If you use Amazon Transcribe multiple Regions, the vocabulary
     * must be available in Amazon Transcribe in each Region.</p> <p>If you include
     * <code>IdentifyLanguage</code> and want to use one or more custom vocabularies
     * with your transcription, use the <code>VocabularyNames</code> parameter
     * instead.</p>
     */
    inline const Aws::String& GetVocabularyName() const { return m_vocabularyName; }
    inline bool VocabularyNameHasBeenSet() const { return m_vocabularyNameHasBeenSet; }
    template<typename VocabularyNameT = Aws::String>
    void SetVocabularyName(VocabularyNameT&& value) { m_vocabularyNameHasBeenSet = true; m_vocabularyName = std::forward<VocabularyNameT>(value); }
    template<typename VocabularyNameT = Aws::String>
    EngineTranscribeSettings& WithVocabularyName(VocabularyNameT&& value) { SetVocabularyName(std::forward<VocabularyNameT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The Amazon Web Services Region in which to use Amazon Transcribe.</p> <p>If
     * you don't specify a Region, then the <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html">MediaRegion</a>
     * of the meeting is used. However, if Amazon Transcribe is not available in the
     * <code>MediaRegion</code>, then a <code>TranscriptFailed</code> event is
     * sent.</p> <p>Use <code>auto</code> to use Amazon Transcribe in a Region near the
     * meeting’s <code>MediaRegion</code>. For more information, refer to <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region">Choosing
     * a transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.</p>
     */
    inline TranscribeRegion GetRegion() const { return m_region; }
    inline bool RegionHasBeenSet() const { return m_regionHasBeenSet; }
    inline void SetRegion(TranscribeRegion value) { m_regionHasBeenSet = true; m_region = value; }
    inline EngineTranscribeSettings& WithRegion(TranscribeRegion value) { SetRegion(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Enables partial result stabilization for your transcription. Partial result
     * stabilization can reduce latency in your output, but may impact accuracy.</p>
     */
    inline bool GetEnablePartialResultsStabilization() const { return m_enablePartialResultsStabilization; }
    inline bool EnablePartialResultsStabilizationHasBeenSet() const { return m_enablePartialResultsStabilizationHasBeenSet; }
    inline void SetEnablePartialResultsStabilization(bool value) { m_enablePartialResultsStabilizationHasBeenSet = true; m_enablePartialResultsStabilization = value; }
    inline EngineTranscribeSettings& WithEnablePartialResultsStabilization(bool value) { SetEnablePartialResultsStabilization(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the level of stability to use when you enable partial results
     * stabilization (<code>EnablePartialResultsStabilization</code>).</p> <p>Low
     * stability provides the highest accuracy. High stability transcribes faster, but
     * with slightly lower accuracy.</p>
     */
    inline TranscribePartialResultsStability GetPartialResultsStability() const { return m_partialResultsStability; }
    inline bool PartialResultsStabilityHasBeenSet() const { return m_partialResultsStabilityHasBeenSet; }
    inline void SetPartialResultsStability(TranscribePartialResultsStability value) { m_partialResultsStabilityHasBeenSet = true; m_partialResultsStability = value; }
    inline EngineTranscribeSettings& WithPartialResultsStability(TranscribePartialResultsStability value) { SetPartialResultsStability(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Labels all personally identifiable information (PII) identified in your
     * transcript. If you don't include <code>PiiEntityTypes</code>, all PII is
     * identified.</p>  <p>You can’t set <code>ContentIdentificationType</code>
     * and <code>ContentRedactionType</code>.</p> 
     */
    inline TranscribeContentIdentificationType GetContentIdentificationType() const { return m_contentIdentificationType; }
    inline bool ContentIdentificationTypeHasBeenSet() const { return m_contentIdentificationTypeHasBeenSet; }
    inline void SetContentIdentificationType(TranscribeContentIdentificationType value) { m_contentIdentificationTypeHasBeenSet = true; m_contentIdentificationType = value; }
    inline EngineTranscribeSettings& WithContentIdentificationType(TranscribeContentIdentificationType value) { SetContentIdentificationType(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Content redaction is performed at the segment level. If you don't include
     * <code>PiiEntityTypes</code>, all PII is redacted.</p>  <p>You can’t set
     * <code>ContentRedactionType</code> and
     * <code>ContentIdentificationType</code>.</p> 
     */
    inline TranscribeContentRedactionType GetContentRedactionType() const { return m_contentRedactionType; }
    inline bool ContentRedactionTypeHasBeenSet() const { return m_contentRedactionTypeHasBeenSet; }
    inline void SetContentRedactionType(TranscribeContentRedactionType value) { m_contentRedactionTypeHasBeenSet = true; m_contentRedactionType = value; }
    inline EngineTranscribeSettings& WithContentRedactionType(TranscribeContentRedactionType value) { SetContentRedactionType(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify which types of personally identifiable information (PII) you want to
     * redact in your transcript. You can include as many types as you'd like, or you
     * can select <code>ALL</code>.</p> <p>Values must be comma-separated and can
     * include: <code>ADDRESS</code>, <code>BANK_ACCOUNT_NUMBER</code>,
     * <code>BANK_ROUTING</code>, <code>CREDIT_DEBIT_CVV</code>,
     * <code>CREDIT_DEBIT_EXPIRY</code> <code>CREDIT_DEBIT_NUMBER</code>,
     * <code>EMAIL</code>,<code>NAME</code>, <code>PHONE</code>, <code>PIN</code>,
     * <code>SSN</code>, or <code>ALL</code>.</p> <p>Note that if you include
     * <code>PiiEntityTypes</code>, you must also include
     * <code>ContentIdentificationType</code> or <code>ContentRedactionType</code>.</p>
     * <p>If you include <code>ContentRedactionType</code> or
     * <code>ContentIdentificationType</code>, but do not include PiiEntityTypes, all
     * PII is redacted or identified.</p>
     */
    inline const Aws::String& GetPiiEntityTypes() const { return m_piiEntityTypes; }
    inline bool PiiEntityTypesHasBeenSet() const { return m_piiEntityTypesHasBeenSet; }
    template<typename PiiEntityTypesT = Aws::String>
    void SetPiiEntityTypes(PiiEntityTypesT&& value) { m_piiEntityTypesHasBeenSet = true; m_piiEntityTypes = std::forward<PiiEntityTypesT>(value); }
    template<typename PiiEntityTypesT = Aws::String>
    EngineTranscribeSettings& WithPiiEntityTypes(PiiEntityTypesT&& value) { SetPiiEntityTypes(std::forward<PiiEntityTypesT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the name of the custom language model that you want to use when
     * processing your transcription. Note that language model names are case
     * sensitive.</p> <p>The language of the specified language model must match the
     * language code. If the languages don't match, the custom language model isn't
     * applied. There are no errors or warnings associated with a language
     * mismatch.</p> <p>If you use Amazon Transcribe in multiple Regions, the custom
     * language model must be available in Amazon Transcribe in each Region.</p>
     */
    inline const Aws::String& GetLanguageModelName() const { return m_languageModelName; }
    inline bool LanguageModelNameHasBeenSet() const { return m_languageModelNameHasBeenSet; }
    template<typename LanguageModelNameT = Aws::String>
    void SetLanguageModelName(LanguageModelNameT&& value) { m_languageModelNameHasBeenSet = true; m_languageModelName = std::forward<LanguageModelNameT>(value); }
    template<typename LanguageModelNameT = Aws::String>
    EngineTranscribeSettings& WithLanguageModelName(LanguageModelNameT&& value) { SetLanguageModelName(std::forward<LanguageModelNameT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Enables automatic language identification for your transcription.</p> <p>If
     * you include <code>IdentifyLanguage</code>, you can optionally use
     * <code>LanguageOptions</code> to include a list of language codes that you think
     * may be present in your audio stream. Including language options can improve
     * transcription accuracy.</p> <p>You can also use <code>PreferredLanguage</code>
     * to include a preferred language. Doing so can help Amazon Transcribe identify
     * the language faster.</p> <p>You must include either <code>LanguageCode</code> or
     * <code>IdentifyLanguage</code>.</p> <p>Language identification can't be combined
     * with custom language models or redaction.</p>
     */
    inline bool GetIdentifyLanguage() const { return m_identifyLanguage; }
    inline bool IdentifyLanguageHasBeenSet() const { return m_identifyLanguageHasBeenSet; }
    inline void SetIdentifyLanguage(bool value) { m_identifyLanguageHasBeenSet = true; m_identifyLanguage = value; }
    inline EngineTranscribeSettings& WithIdentifyLanguage(bool value) { SetIdentifyLanguage(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify two or more language codes that represent the languages you think may
     * be present in your media; including more than five is not recommended. If you're
     * unsure what languages are present, do not include this parameter.</p>
     * <p>Including language options can improve the accuracy of language
     * identification.</p> <p>If you include <code>LanguageOptions</code>, you must
     * also include <code>IdentifyLanguage</code>.</p>  <p>You can only
     * include one language dialect per language. For example, you cannot include
     * <code>en-US</code> and <code>en-AU</code>.</p> 
     */
    inline const Aws::String& GetLanguageOptions() const { return m_languageOptions; }
    inline bool LanguageOptionsHasBeenSet() const { return m_languageOptionsHasBeenSet; }
    template<typename LanguageOptionsT = Aws::String>
    void SetLanguageOptions(LanguageOptionsT&& value) { m_languageOptionsHasBeenSet = true; m_languageOptions = std::forward<LanguageOptionsT>(value); }
    template<typename LanguageOptionsT = Aws::String>
    EngineTranscribeSettings& WithLanguageOptions(LanguageOptionsT&& value) { SetLanguageOptions(std::forward<LanguageOptionsT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify a preferred language from the subset of languages codes you specified
     * in <code>LanguageOptions</code>.</p> <p>You can only use this parameter if you
     * include <code>IdentifyLanguage</code> and <code>LanguageOptions</code>.</p>
     */
    inline TranscribeLanguageCode GetPreferredLanguage() const { return m_preferredLanguage; }
    inline bool PreferredLanguageHasBeenSet() const { return m_preferredLanguageHasBeenSet; }
    inline void SetPreferredLanguage(TranscribeLanguageCode value) { m_preferredLanguageHasBeenSet = true; m_preferredLanguage = value; }
    inline EngineTranscribeSettings& WithPreferredLanguage(TranscribeLanguageCode value) { SetPreferredLanguage(value); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the names of the custom vocabularies that you want to use when
     * processing your transcription. Note that vocabulary names are case
     * sensitive.</p> <p>If you use Amazon Transcribe in multiple Regions, the
     * vocabulary must be available in Amazon Transcribe in each Region.</p> <p>If you
     * don't include <code>IdentifyLanguage</code> and want to use a custom vocabulary
     * with your transcription, use the <code>VocabularyName</code> parameter
     * instead.</p>
     */
    inline const Aws::String& GetVocabularyNames() const { return m_vocabularyNames; }
    inline bool VocabularyNamesHasBeenSet() const { return m_vocabularyNamesHasBeenSet; }
    template<typename VocabularyNamesT = Aws::String>
    void SetVocabularyNames(VocabularyNamesT&& value) { m_vocabularyNamesHasBeenSet = true; m_vocabularyNames = std::forward<VocabularyNamesT>(value); }
    template<typename VocabularyNamesT = Aws::String>
    EngineTranscribeSettings& WithVocabularyNames(VocabularyNamesT&& value) { SetVocabularyNames(std::forward<VocabularyNamesT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>Specify the names of the custom vocabulary filters that you want to use when
     * processing your transcription. Note that vocabulary filter names are case
     * sensitive.</p> <p>If you use Amazon Transcribe in multiple Regions, the
     * vocabulary filter must be available in Amazon Transcribe in each Region.</p> <p>
     * If you're <i>not</i> including <code>IdentifyLanguage</code> and want to use a
     * custom vocabulary filter with your transcription, use the
     * <code>VocabularyFilterName</code> parameter instead.</p>
     */
    inline const Aws::String& GetVocabularyFilterNames() const { return m_vocabularyFilterNames; }
    inline bool VocabularyFilterNamesHasBeenSet() const { return m_vocabularyFilterNamesHasBeenSet; }
    template<typename VocabularyFilterNamesT = Aws::String>
    void SetVocabularyFilterNames(VocabularyFilterNamesT&& value) { m_vocabularyFilterNamesHasBeenSet = true; m_vocabularyFilterNames = std::forward<VocabularyFilterNamesT>(value); }
    template<typename VocabularyFilterNamesT = Aws::String>
    EngineTranscribeSettings& WithVocabularyFilterNames(VocabularyFilterNamesT&& value) { SetVocabularyFilterNames(std::forward<VocabularyFilterNamesT>(value)); return *this;}
    ///@}
  private:

    TranscribeLanguageCode m_languageCode{TranscribeLanguageCode::NOT_SET};
    bool m_languageCodeHasBeenSet = false;

    TranscribeVocabularyFilterMethod m_vocabularyFilterMethod{TranscribeVocabularyFilterMethod::NOT_SET};
    bool m_vocabularyFilterMethodHasBeenSet = false;

    Aws::String m_vocabularyFilterName;
    bool m_vocabularyFilterNameHasBeenSet = false;

    Aws::String m_vocabularyName;
    bool m_vocabularyNameHasBeenSet = false;

    TranscribeRegion m_region{TranscribeRegion::NOT_SET};
    bool m_regionHasBeenSet = false;

    bool m_enablePartialResultsStabilization{false};
    bool m_enablePartialResultsStabilizationHasBeenSet = false;

    TranscribePartialResultsStability m_partialResultsStability{TranscribePartialResultsStability::NOT_SET};
    bool m_partialResultsStabilityHasBeenSet = false;

    TranscribeContentIdentificationType m_contentIdentificationType{TranscribeContentIdentificationType::NOT_SET};
    bool m_contentIdentificationTypeHasBeenSet = false;

    TranscribeContentRedactionType m_contentRedactionType{TranscribeContentRedactionType::NOT_SET};
    bool m_contentRedactionTypeHasBeenSet = false;

    Aws::String m_piiEntityTypes;
    bool m_piiEntityTypesHasBeenSet = false;

    Aws::String m_languageModelName;
    bool m_languageModelNameHasBeenSet = false;

    bool m_identifyLanguage{false};
    bool m_identifyLanguageHasBeenSet = false;

    Aws::String m_languageOptions;
    bool m_languageOptionsHasBeenSet = false;

    TranscribeLanguageCode m_preferredLanguage{TranscribeLanguageCode::NOT_SET};
    bool m_preferredLanguageHasBeenSet = false;

    Aws::String m_vocabularyNames;
    bool m_vocabularyNamesHasBeenSet = false;

    Aws::String m_vocabularyFilterNames;
    bool m_vocabularyFilterNamesHasBeenSet = false;
  };

} // namespace Model
} // namespace ChimeSDKMeetings
} // namespace Aws
