/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.common.time;

import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.TemporalQueries;
import java.util.Objects;
import java.util.function.LongSupplier;
import org.opensearch.OpenSearchParseException;
import org.opensearch.common.time.DateFormatters;
import org.opensearch.common.time.DateMathParser;
import org.opensearch.common.time.JavaDateFormatter;
import org.opensearch.core.common.Strings;

public class JavaDateMathParser
implements DateMathParser {
    private final JavaDateFormatter formatter;
    private final String format;
    private final JavaDateFormatter roundupParser;

    JavaDateMathParser(String format, JavaDateFormatter formatter, JavaDateFormatter roundupParser) {
        this.format = format;
        this.roundupParser = roundupParser;
        Objects.requireNonNull(formatter);
        this.formatter = formatter;
    }

    @Override
    public Instant parse(String text, LongSupplier now, boolean roundUpProperty, ZoneId timeZone) {
        String mathString;
        Instant time;
        if (text.startsWith("now")) {
            try {
                time = Instant.ofEpochMilli(now.getAsLong());
            }
            catch (Exception e) {
                throw new OpenSearchParseException("could not read the current timestamp", (Throwable)e, new Object[0]);
            }
            mathString = text.substring("now".length());
        } else {
            int index = text.indexOf("||");
            if (index == -1) {
                return this.parseDateTime(text, timeZone, roundUpProperty);
            }
            time = this.parseDateTime(text.substring(0, index), timeZone, false);
            mathString = text.substring(index + 2);
        }
        return this.parseMath(mathString, time, roundUpProperty, timeZone);
    }

    private Instant parseMath(String mathString, Instant time, boolean roundUpProperty, ZoneId timeZone) throws OpenSearchParseException {
        if (timeZone == null) {
            timeZone = ZoneOffset.UTC;
        }
        ZonedDateTime dateTime = ZonedDateTime.ofInstant(time, timeZone);
        int i = 0;
        while (i < mathString.length()) {
            int num;
            int sign;
            boolean round;
            char c;
            if ((c = mathString.charAt(i++)) == '/') {
                round = true;
                sign = 1;
            } else {
                round = false;
                if (c == '+') {
                    sign = 1;
                } else if (c == '-') {
                    sign = -1;
                } else {
                    throw new OpenSearchParseException("operator not supported for date math [{}]", new Object[]{mathString});
                }
            }
            if (i >= mathString.length()) {
                throw new OpenSearchParseException("truncated date math [{}]", new Object[]{mathString});
            }
            if (!Character.isDigit(mathString.charAt(i))) {
                num = 1;
            } else {
                int numFrom = i;
                while (i < mathString.length() && Character.isDigit(mathString.charAt(i))) {
                    ++i;
                }
                if (i >= mathString.length()) {
                    throw new OpenSearchParseException("truncated date math [{}]", new Object[]{mathString});
                }
                num = Integer.parseInt(mathString.substring(numFrom, i));
            }
            if (round && num != 1) {
                throw new OpenSearchParseException("rounding `/` can only be used on single unit types [{}]", new Object[]{mathString});
            }
            char unit = mathString.charAt(i++);
            switch (unit) {
                case 'y': {
                    if (round) {
                        dateTime = dateTime.withDayOfYear(1).with(LocalTime.MIN);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusYears(1L);
                        break;
                    }
                    dateTime = dateTime.plusYears(sign * num);
                    break;
                }
                case 'M': {
                    if (round) {
                        dateTime = dateTime.withDayOfMonth(1).with(LocalTime.MIN);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusMonths(1L);
                        break;
                    }
                    dateTime = dateTime.plusMonths(sign * num);
                    break;
                }
                case 'w': {
                    if (round) {
                        dateTime = dateTime.with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY)).with(LocalTime.MIN);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusWeeks(1L);
                        break;
                    }
                    dateTime = dateTime.plusWeeks(sign * num);
                    break;
                }
                case 'd': {
                    if (round) {
                        dateTime = dateTime.with(LocalTime.MIN);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusDays(1L);
                        break;
                    }
                    dateTime = dateTime.plusDays(sign * num);
                    break;
                }
                case 'H': 
                case 'h': {
                    if (round) {
                        dateTime = dateTime.withMinute(0).withSecond(0).withNano(0);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusHours(1L);
                        break;
                    }
                    dateTime = dateTime.plusHours(sign * num);
                    break;
                }
                case 'm': {
                    if (round) {
                        dateTime = dateTime.withSecond(0).withNano(0);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusMinutes(1L);
                        break;
                    }
                    dateTime = dateTime.plusMinutes(sign * num);
                    break;
                }
                case 's': {
                    if (round) {
                        dateTime = dateTime.withNano(0);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusSeconds(1L);
                        break;
                    }
                    dateTime = dateTime.plusSeconds(sign * num);
                    break;
                }
                default: {
                    throw new OpenSearchParseException("unit [{}] not supported for date math [{}]", new Object[]{Character.valueOf(unit), mathString});
                }
            }
            if (!round || !roundUpProperty) continue;
            dateTime = dateTime.minus(1L, ChronoField.MILLI_OF_SECOND.getBaseUnit());
        }
        return dateTime.toInstant();
    }

    private Instant parseDateTime(String value, ZoneId timeZone, boolean roundUpIfNoTime) {
        if (Strings.isNullOrEmpty((String)value)) {
            throw new OpenSearchParseException("cannot parse empty date", new Object[0]);
        }
        JavaDateFormatter formatter = roundUpIfNoTime ? this.roundupParser : this.formatter;
        try {
            if (timeZone == null) {
                return DateFormatters.from(formatter.parse(value)).toInstant();
            }
            TemporalAccessor accessor = formatter.parse(value);
            ZoneId zoneId = TemporalQueries.zone().queryFrom(accessor);
            if (zoneId != null) {
                timeZone = zoneId;
            }
            return DateFormatters.from(accessor).withZoneSameLocal(timeZone).toInstant();
        }
        catch (IllegalArgumentException | DateTimeParseException e) {
            throw new OpenSearchParseException("failed to parse date field [{}] with format [{}]: [{}]", (Throwable)e, new Object[]{value, this.format, e.getMessage()});
        }
    }
}

