"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeAlgorithmSuite = void 0;
/*
 * This file contains information about particular algorithm suites used
 * within the encryption SDK.  In most cases, end-users don't need to
 * manipulate this structure, but it can occasionally be needed for more
 * advanced use cases, such as writing keyrings.
 *
 * These are the Node.js specific values the AWS Encryption SDK for JavaScript
 * Algorithm Suites.
 */
const algorithm_suites_1 = require("./algorithm_suites");
/* References to https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/algorithms-reference.html
 * These are the composed parameters for each algorithm suite specification for
 * for the WebCrypto environment.
 */
const nodeAlgAes128GcmIv12Tag16 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'aes-128-gcm',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const nodeAlgAes192GcmIv12Tag16 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'aes-192-gcm',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const nodeAlgAes256GcmIv12Tag16 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    cacheSafe: false,
    commitment: 'NONE',
};
const nodeAlgAes128GcmIv12Tag16HkdfSha256 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'aes-128-gcm',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha256',
    cacheSafe: true,
    commitment: 'NONE',
};
const nodeAlgAes192GcmIv12Tag16HkdfSha256 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'aes-192-gcm',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha256',
    cacheSafe: true,
    commitment: 'NONE',
};
const nodeAlgAes256GcmIv12Tag16HkdfSha256 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha256',
    cacheSafe: true,
    commitment: 'NONE',
};
const nodeAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'aes-128-gcm',
    keyLength: 128,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha256',
    cacheSafe: true,
    signatureCurve: 'prime256v1',
    signatureHash: 'sha256',
    commitment: 'NONE',
};
const nodeAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'aes-192-gcm',
    keyLength: 192,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha384',
    cacheSafe: true,
    signatureCurve: 'secp384r1',
    signatureHash: 'sha384',
    commitment: 'NONE',
};
const nodeAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384,
    messageFormat: algorithm_suites_1.MessageFormat.V1,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha384',
    cacheSafe: true,
    signatureCurve: 'secp384r1',
    signatureHash: 'sha384',
    commitment: 'NONE',
};
const nodeAlgAes256GcmHkdfSha512Committing = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY,
    messageFormat: algorithm_suites_1.MessageFormat.V2,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha512',
    cacheSafe: true,
    commitment: 'KEY',
    commitmentHash: 'sha512',
    suiteDataLength: 32,
    commitmentLength: 256,
    saltLengthBytes: 32,
};
const nodeAlgAes256GcmHkdfSha512CommittingEcdsaP384 = {
    id: algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384,
    messageFormat: algorithm_suites_1.MessageFormat.V2,
    encryption: 'aes-256-gcm',
    keyLength: 256,
    ivLength: 12,
    tagLength: 128,
    kdf: 'HKDF',
    kdfHash: 'sha512',
    cacheSafe: true,
    signatureCurve: 'secp384r1',
    signatureHash: 'sha384',
    commitment: 'KEY',
    commitmentHash: 'sha512',
    suiteDataLength: 32,
    commitmentLength: 256,
    saltLengthBytes: 32,
};
const nodeAlgorithms = Object.freeze({
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16]: Object.freeze(nodeAlgAes128GcmIv12Tag16),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16]: Object.freeze(nodeAlgAes192GcmIv12Tag16),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16]: Object.freeze(nodeAlgAes256GcmIv12Tag16),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(nodeAlgAes128GcmIv12Tag16HkdfSha256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(nodeAlgAes192GcmIv12Tag16HkdfSha256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA256]: Object.freeze(nodeAlgAes256GcmIv12Tag16HkdfSha256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES128_GCM_IV12_TAG16_HKDF_SHA256_ECDSA_P256]: Object.freeze(nodeAlgAes128GcmIv12Tag16HkdfSha256EcdsaP256),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES192_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(nodeAlgAes192GcmIv12Tag16HkdfSha384EcdsaP384),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA384_ECDSA_P384]: Object.freeze(nodeAlgAes256GcmIv12Tag16HkdfSha384EcdsaP384),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY]: Object.freeze(nodeAlgAes256GcmHkdfSha512Committing),
    [algorithm_suites_1.AlgorithmSuiteIdentifier.ALG_AES256_GCM_IV12_TAG16_HKDF_SHA512_COMMIT_KEY_ECDSA_P384]: Object.freeze(nodeAlgAes256GcmHkdfSha512CommittingEcdsaP384),
});
class NodeAlgorithmSuite extends algorithm_suites_1.AlgorithmSuite {
    type = 'node';
    constructor(id) {
        super(nodeAlgorithms[id]);
        Object.setPrototypeOf(this, NodeAlgorithmSuite.prototype);
        Object.freeze(this);
    }
}
exports.NodeAlgorithmSuite = NodeAlgorithmSuite;
Object.freeze(NodeAlgorithmSuite.prototype);
Object.freeze(NodeAlgorithmSuite);
//# sourceMappingURL=data:application/json;base64,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