"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JSONProcessor = void 0;
var _util = require("../utils/util");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

class JSONProcessor {
  async validateText(text, _) {
    if (text.length < 1) {
      return false;
    }
    try {
      const obj = JSON.parse(text);
      return obj && typeof obj === 'object' && (0, _util.isValidObject)(obj);
    } catch (e) {
      return false;
    }
  }
  async ingestText(text, options) {
    const {
      client,
      indexName
    } = options;
    const document = JSON.parse(text);
    const isSuccessful = await new Promise(async resolve => {
      try {
        await client.index({
          index: indexName,
          body: document
        });
        resolve(true);
      } catch (e) {
        resolve(false);
      }
    });
    const total = isSuccessful ? 1 : 0;
    return {
      total: 1,
      message: `Indexed ${total} document`,
      failedRows: isSuccessful ? [] : [1]
    };
  }
  async ingestFile(file, options) {
    const {
      client,
      indexName
    } = options;
    const numSucessfulDocs = await new Promise(resolve => {
      let rawData = '';
      file.on('data', chunk => rawData += chunk).on('error', _ => resolve(0)).on('end', async () => {
        try {
          const document = JSON.parse(rawData);
          if (!(0, _util.isValidObject)(document)) {
            resolve(0);
          }
          await client.index({
            index: indexName,
            body: document
          });
          resolve(1);
        } catch (_) {
          resolve(0);
        }
      });
    });
    return {
      total: 1,
      message: `Indexed ${numSucessfulDocs} document`,
      failedRows: numSucessfulDocs === 1 ? [] : [1]
    };
  }
  async parseFile(file, limit, _) {
    const documents = [];
    await new Promise((resolve, reject) => {
      let rawData = '';
      file.on('data', chunk => rawData += chunk).on('error', e => reject(e)).on('end', async () => {
        try {
          const document = JSON.parse(rawData);
          if (!(0, _util.isValidObject)(document)) {
            reject(new Error(`The following document has empty fields: ${JSON.stringify(document)}`));
          }
          documents.push(document);
        } catch (e) {
          reject(e);
        }
        resolve();
      });
    });
    return documents;
  }
}
exports.JSONProcessor = JSONProcessor;