/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.cluster.maintenance;

import com.google.common.annotations.VisibleForTesting;
import java.lang.invoke.MethodHandles;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.solr.api.ConfigurablePlugin;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrResponse;
import org.apache.solr.client.solrj.request.CollectionAdminRequest;
import org.apache.solr.cloud.ClusterSingleton;
import org.apache.solr.cluster.maintenance.InactiveShardRemoverConfig;
import org.apache.solr.common.cloud.ClusterState;
import org.apache.solr.common.cloud.DocCollection;
import org.apache.solr.common.cloud.Slice;
import org.apache.solr.common.util.ExecutorUtil;
import org.apache.solr.common.util.SolrNamedThreadFactory;
import org.apache.solr.core.CoreContainer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InactiveShardRemover
implements ClusterSingleton,
ConfigurablePlugin<InactiveShardRemoverConfig> {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    public static final String PLUGIN_NAME = ".inactive-shard-remover";
    private ClusterSingleton.State state = ClusterSingleton.State.STOPPED;
    private final CoreContainer coreContainer;
    private final DeleteActor deleteActor;
    private ScheduledExecutorService executor;
    private long scheduleIntervalSeconds;
    private long ttlSeconds;
    private int maxDeletesPerCycle;

    public InactiveShardRemover(CoreContainer cc) {
        this(cc, new DeleteActor(cc));
    }

    public InactiveShardRemover(CoreContainer cc, DeleteActor deleteActor) {
        this.coreContainer = cc;
        this.deleteActor = deleteActor;
    }

    @Override
    public void configure(InactiveShardRemoverConfig cfg) {
        Objects.requireNonNull(cfg, "config must be specified");
        cfg.validate();
        this.scheduleIntervalSeconds = cfg.scheduleIntervalSeconds;
        this.maxDeletesPerCycle = cfg.maxDeletesPerCycle;
        this.ttlSeconds = cfg.ttlSeconds;
    }

    @Override
    public String getName() {
        return PLUGIN_NAME;
    }

    @Override
    public ClusterSingleton.State getState() {
        return this.state;
    }

    @Override
    public void start() throws Exception {
        this.state = ClusterSingleton.State.STARTING;
        this.executor = Executors.newScheduledThreadPool(1, (ThreadFactory)new SolrNamedThreadFactory(PLUGIN_NAME));
        this.executor.scheduleAtFixedRate(this::deleteInactiveSlices, this.scheduleIntervalSeconds, this.scheduleIntervalSeconds, TimeUnit.SECONDS);
        this.state = ClusterSingleton.State.RUNNING;
    }

    @Override
    public void stop() {
        if (this.state == ClusterSingleton.State.RUNNING) {
            this.state = ClusterSingleton.State.STOPPING;
            ExecutorUtil.shutdownNowAndAwaitTermination((ExecutorService)this.executor);
        }
        this.state = ClusterSingleton.State.STOPPED;
    }

    @VisibleForTesting
    void deleteInactiveSlices() {
        ClusterState clusterState = this.coreContainer.getZkController().getClusterState();
        Collection inactiveSlices = clusterState.collectionStream().flatMap(v -> this.collectInactiveSlices((DocCollection)v).stream()).collect(Collectors.toSet());
        if (log.isInfoEnabled()) {
            log.info("Found {} inactive Shards to delete, {} will be deleted", (Object)inactiveSlices.size(), (Object)Math.min(inactiveSlices.size(), this.maxDeletesPerCycle));
        }
        inactiveSlices.stream().limit(this.maxDeletesPerCycle).forEach(this::deleteShard);
    }

    private Collection<Slice> collectInactiveSlices(DocCollection docCollection) {
        HashSet slices = new HashSet(docCollection.getSlices());
        slices.removeAll(docCollection.getActiveSlices());
        return slices.stream().filter(this::isExpired).collect(Collectors.toSet());
    }

    private void deleteShard(Slice s) {
        this.deleteActor.delete(s);
    }

    private boolean isExpired(Slice slice) {
        boolean expired;
        long epochTimestampNs;
        String collectionName = slice.getCollection();
        String sliceName = slice.getName();
        if (slice.getState() != Slice.State.INACTIVE) {
            return false;
        }
        String lastChangeTimestamp = slice.getStr("stateTimestamp");
        if (lastChangeTimestamp == null || lastChangeTimestamp.isEmpty()) {
            log.warn("Collection {} Shard {} has no last change timestamp and will not be deleted", (Object)collectionName, (Object)sliceName);
            return false;
        }
        try {
            epochTimestampNs = Long.parseLong(lastChangeTimestamp);
        }
        catch (NumberFormatException e) {
            log.warn("Collection {} Shard {} has an invalid last change timestamp and will not be deleted", (Object)collectionName, (Object)sliceName);
            return false;
        }
        long currentEpochTimeNs = this.coreContainer.getZkController().getSolrCloudManager().getTimeSource().getEpochTimeNs();
        long delta = TimeUnit.NANOSECONDS.toSeconds(currentEpochTimeNs - epochTimestampNs);
        boolean bl = expired = delta >= this.ttlSeconds;
        if (log.isDebugEnabled()) {
            log.debug("collection {} shard {} last state change {} seconds ago. Expired={}", new Object[]{slice.getCollection(), slice.getName(), delta, expired});
        }
        return expired;
    }

    static class DeleteActor {
        private final CoreContainer coreContainer;

        DeleteActor(CoreContainer coreContainer) {
            this.coreContainer = coreContainer;
        }

        void delete(Slice slice) {
            CollectionAdminRequest.DeleteShard deleteRequest = CollectionAdminRequest.deleteShard((String)slice.getCollection(), (String)slice.getName());
            try {
                SolrResponse response = deleteRequest.process((SolrClient)this.coreContainer.getZkController().getSolrClient());
                if (response.getException() != null) {
                    throw response.getException();
                }
            }
            catch (Exception e) {
                log.warn("An exception occurred when deleting an inactive shard", (Throwable)e);
            }
        }
    }
}

